package com.vmware.vide.vlogbrowser.service;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.zip.GZIPOutputStream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.DefaultLogBrowserManager;
import com.vmware.vide.vlogbrowser.core.ILogBrowserManager;
import com.vmware.vide.vlogbrowser.core.chart.IFilterListener;
import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.model.FilterItem;
import com.vmware.vide.vlogbrowser.core.model.FilterList;
import com.vmware.vide.vlogbrowser.core.model.ILogItemList;
import com.vmware.vide.vlogbrowser.core.model.LogDate;
import com.vmware.vide.vlogbrowser.core.model.LogFilter;
import com.vmware.vide.vlogbrowser.core.model.LogItem;
import com.vmware.vide.vlogbrowser.core.parser.LogFormat;
import com.vmware.vide.vlogbrowser.core.parser.RAFile;
import com.vmware.vide.vlogbrowser.core.utils.Preferences;
import com.vmware.vide.vlogbrowser.service.model.FilterConditionResult;
import com.vmware.vide.vlogbrowser.service.model.FilterConditionRowResult;
import com.vmware.vide.vlogbrowser.service.model.FilterResult;
import com.vmware.vide.vlogbrowser.service.model.LogEntries;
import com.vmware.vide.vlogbrowser.service.model.LogEntryResult;
import com.vmware.vide.vlogbrowser.service.model.LogFileResult;
import com.vmware.vide.vlogbrowser.service.model.LogTableResult;
import com.vmware.vide.vlogbrowser.service.model.ProgressResult;
import com.vmware.vide.vlogbrowser.service.model.ProgressResult.ProgressStatus;
import com.vmware.vide.vlogbrowser.service.model.ProjectManifest;
import com.vmware.vide.vlogbrowser.service.model.ResourceInfo;

public class LogBrowserFacade implements ILogBrowserService, Observer {

    private ILogBrowserManager logManager;
    private static LogProjectManager prjManager;
    private String publicFolder;

    private Map<String, ProgressResult> mapResnameProgress = new HashMap<String, ProgressResult>();

    private static final long CLEANUP_BUNDLES_MILLISECONDS = 7 * 24 * 60 * 60 * 1000; // 7 days
    private static final String PROP_XPATHFACTORY = "javax.xml.xpath.XPathFactory:http://java.sun.com/jaxp/xpath/dom";

    private static final Logger logger = LoggerFactory.getLogger(LogBrowserFacade.class);

    public LogBrowserFacade(String rootPath) {
        Preferences prefs = Preferences.getInstance();
        prefs.put(Preferences.ROOT_DIR, rootPath);

        File dir = new File(rootPath, LogBrowserConsts.PUBLIC_FOLDER_DIR);
        publicFolder = dir.getAbsolutePath();
        dir.mkdirs();

        logManager = new DefaultLogBrowserManager();
        prjManager = new LogProjectManager(rootPath);

        // Sets the XPathFactory provider if not already set.
        if (System.getProperty(PROP_XPATHFACTORY) == null) {
            System.setProperty(PROP_XPATHFACTORY, "com.sun.org.apache.xpath.internal.jaxp.XPathFactoryImpl");
        }
    }

    @Override
    public List<ProjectManifest> getProjectManifest() throws Exception {
        List<ProjectManifest> result = new ArrayList<ProjectManifest>();
        ProjectManifest entry = null;
        String prjName = LogProjectManager.PROJECT_DEFAULT_NAME;
        File prj = prjManager.find(prjName);
        if (prj == null) {
            throw new Exception("Project '" +
                    prjName + "' does not exist.");
        }
        
        String[] files = prj.list();
        for (String manifestName: files) {
            if (!manifestName.endsWith("_manifest.txt")) {
                continue;
            }

            BufferedReader br = null;
            try {
                String fullPath = prj.getAbsolutePath() + File.separator + manifestName;
                FileInputStream reader = new FileInputStream(fullPath);
                // Get the object of DataInputStream
                DataInputStream in = new DataInputStream(reader);
                br = new BufferedReader(new InputStreamReader(in));
                String strLine;

                String dateFetched = "Never";
                String bundleName = "";
                String sourceName = "";
                List<LogFileResult> logFiles = new ArrayList<LogFileResult>();
                LogFileResult logFile = null;

                //Read File Line By Line
                while ((strLine = br.readLine()) != null)   {
                    if (strLine.startsWith("file=")) {
                        logFile = new LogFileResult();
                        logFiles.add(logFile);
                        logFile.setName(strLine.substring(5));
                    } else if (strLine.startsWith("date=")) {
                        if (logFile != null) {
                            logFile.setDate(strLine.substring(5));
                        }
                        dateFetched = strLine.substring(5);
                    } else if (strLine.startsWith("type=")) {
                        if(logFile != null) {
                            logFile.setType(strLine.substring(5));
                        }
                    } else if (strLine.startsWith("bundle=")) {
                        bundleName = strLine.substring(7);
                    } else if (strLine.startsWith("source=")) {
                        sourceName = strLine.substring(7);
                    }
                }
                entry = new ProjectManifest();
                entry.setDate(dateFetched);
                entry.setSource(sourceName);
                entry.setBundle(bundleName);
                LogFileResult[] array = logFiles.toArray(new LogFileResult[logFiles.size()]);
                entry.setLogFiles(array);
                File bundleFile = new File(bundleName);
                if (bundleFile.exists()) {
                    entry.setBundleSize(bundleFile.length());
                }
                result.add(entry);
            } finally {
                if (br != null) {
                    br.close();
                }
            }
        }
        return result;
    }

    @Override
    public ProgressResult requestProgress(String resName) throws Exception {
        ProgressResult result = mapResnameProgress.get(resName);
        if (result != null && result.getStatus() == ProgressStatus.ERROR) {
            throw new Exception("Error donwloading bundle!");
        }
        return result;
    }

    @Override
    public ProgressResult refreshLogs(String resName, ResourceInfo info)
            throws Exception {
        ProgressResult result = mapResnameProgress.get(resName);
        if (result == null || result.getStatus() == ProgressStatus.DONE) {
            LogRefreshRunnable task = createThreadRefreshLogs(resName, info);
            result = task.getProgressResult();
            mapResnameProgress.put(resName, result);
        }
        return result;
    }

    private LogRefreshRunnable createThreadRefreshLogs(String resName,
            ResourceInfo info) throws Exception {
        LogRefreshRunnable task = null;

        String host = info.getHost().trim();
        String url = info.getUrl().trim();
        String ticket = info.getTicket().trim();

        String destLocation = getVMSupportFileName(host);
        if (destLocation == null) {
            throw new Exception(
                    "Error creating downloading log bundle: no destination for " + host);
        }
        DefaultHttpClient httpclient = new DefaultHttpClient();
        httpclient = WebClientDevWrapper.wrapClient(httpclient);
        HttpResponse response;
        HttpGet httpget = new HttpGet(url);
        if (ticket != null && ticket.length() > 0) {
            httpget.addHeader("Cookie", "vmware_cgi_ticket=" + ticket);
        }
        response = httpclient.execute(httpget);
        HttpEntity entity = response.getEntity();
        if (entity != null) {
            // delete old tgz, old logx, old cached files
            String manifestName = getManifestFileName(host);
            File manifest = new File(manifestName);
            if (manifest.exists()) {
                deleteAllLogsFromHost(host);
            }

            // Deletes log bundles older than 30 days.
            cleanupOldLogBundles();

            // create thread
            task = new LogRefreshRunnable(entity, destLocation, manifestName, host, resName);
            task.addObserver(this);
//            TaskScheduler.add(task);
            Thread worker = new Thread(task);
            worker.start();
        }
        return task;
    }

    private void cleanupOldLogBundles() {
        File bundleDir = new File(publicFolder);
        if (bundleDir.exists()) {
            for (File bundle: bundleDir.listFiles()) {
                long diff = new Date().getTime() - bundle.lastModified();
                if (diff > CLEANUP_BUNDLES_MILLISECONDS) {
                    bundle.delete();
                }
            }
        }
    }

    @Override
    public Integer getCount(String resName) throws Exception {
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        FilterList items = logFile.getFilterList();
        return items.size();
    }

    @Override
    public LogEntries getEntries(String resName, String filterString,
            int firstIndex, int count) throws Exception {
        if (filterString == null) {
            filterString = "";
        }
        LogEntries res = new LogEntries();
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        res.setStatus(LogTableResult.DONE);

        FilterList filterList;
        ILogItemList items;
        String fltr = filterString; //viewer.getTxtFilterStr();
        filterList = logFile.getFilterList();
        boolean isFiltered = (fltr != null && !fltr.isEmpty());
        if (!isFiltered) {
            items = filterList.getUnfilteredItems();
        } else {
            items = filterList.getFilteredItems();
        }
        if (firstIndex >= items.size()) {
            firstIndex = 0;
        }
        int nEntries = count;
        if (nEntries < 1) {
            nEntries = 30;
        }
        if (firstIndex + nEntries >= items.size()) {
            nEntries = items.size() - firstIndex;
        }
        LogFormat fmt = logFile.getLogFormat();
        String[] cols = fmt.getColumnNames();
        List<LogEntryResult> entries = new ArrayList<LogEntryResult>();
        for (int i = firstIndex ; i < firstIndex + nEntries; ++i) {
            LogItem item = items.get(i);
            LogEntryResult entry = new LogEntryResult();
            entry.setTime(item.getLogDate().toString());
            String[] values = new String[cols.length];
            for(int c = 0; c < cols.length; ++c) {
                try {
                    Object o = item.getLogField(c, fmt);
                    if((o instanceof String)) {
                        String v = (String)o;
                        values[c] = v;
                    } else if ((o instanceof Long)) {
                        String v = ((Long)o).toString();
                        values[c] = v;
                    } else if ((o instanceof LogDate)) {
                        LogDate d = (LogDate) o;
                        values[c] = d.toString();
                        res.setDateDelta(d.getLogDateFormat().getDateDelta());
                    } else if ((o instanceof Date)) {
                        Date date = (Date) o;
                        LogDate d = new LogDate(date.getTime(), fmt.getDateFormat(),0);
                        values[c] = d.toString();
                        res.setDateDelta(d.getLogDateFormat().getDateDelta());
                        entry.setTime(d.toString());
                    }
//                    values[c] = cleanResult(values[c]);
                } catch (IOException e) {
                    values[c] = e.getMessage();
                }
            }
            entry.setValues(values);
            try {
                String extraLines = item.getLogExtra(fmt);
                entry.setExtraLines(extraLines);
            } catch (IOException e) {
                throw new Exception("Unable to retrieve extra info from log entry: " + item, e);
            }
            if (isFiltered)
                entry.setGroup("" + (item.getGroupId() & 1));
            entries.add(entry);
        }
        res.setValues(entries);
        res.setColumns(cols);
        return res;

    }
    
    @Override
    public Integer[] findIndexByDateTime(String resName, int firstIndex, Date dateTime) throws Exception{
        return findPattern(resName, "", firstIndex, dateTime);
    }
    
    @Override
    public Integer[] findIndexByText(String resName, int firstIndex, String pattern) throws Exception{
        return findPattern(resName, pattern, firstIndex, null);
    }
    
    private Integer[] findPattern(
                        String resName,
                        String pattern,
                        int firstIndex,
                        Date time
                        ) throws Exception {
                    LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
                    if (logFile == null) {
                        throw new Exception("Error when try to get an instance of LogFileAnalyser");
                    }
                    LogTableResult result = new LogTableResult();
                    List<Integer> rows = new ArrayList<Integer>();

                    if(time != null) {
                             int row = logFile.findClosestDate(time);
                             rows.add(row);
                    }else {
                        rows = logFile.findPattern(pattern, firstIndex);
                    }
                    Integer[] rowsArray = new Integer[rows.size()];
                    rows.toArray(rowsArray);
                    return rowsArray;
                }

    @Override
    public void adjustTime(String resName, long dateDelta) throws Exception {
        if (resName != null && !resName.isEmpty()) {
            LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
            if (logFile == null) {
                throw new Exception("Error when try to get an instance of LogFileAnalyser");
            }
            LogFormat logFormat = logFile.getLogFormat();
            logFormat.setDateDelta(dateDelta);
        }
    }

    @Override
    public List<FilterResult> getFilters(String resName) throws Exception {
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        LogTableResult result = new LogTableResult();
        result.setFilters(new ArrayList<FilterResult>());
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        Map<String, LogFilter> filters = logFile.getNamedFilterList();
        Set<Entry<String, LogFilter>> names = filters.entrySet();
        Iterator<Entry<String, LogFilter>> x = names.iterator();
        while (x.hasNext()) {
            Entry<String, LogFilter> e = x.next();
            result.addFilter(e.getKey(), e.getValue());
        }
        return result.getFilters();
    }

    @Override
    public FilterResult getFilters(String resName, String filterName)
            throws Exception {
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        Map<String, LogFilter> filters = logFile.getNamedFilterList();
        LogFilter filter = filters.get(filterName);
        if (filter == null) {
            logger.debug("Filter '" + filterName + "' not found.");
            return null;
        }
        FilterResult result = new FilterResult();
        result.setFilterName(filterName);
        result.setFilter(filter);
        return result;
    }

    @Override
    public void applyFilter(String resName, String filterString,
            int aroundString) throws Exception {
        if (filterString == null) {
            filterString = "";
        }
        final LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        String value = stripQuotes(filterString);
        final LogFilter filter;

        if (aroundString == -1) {
            filter = logFile.useLogFilter(value);
        } else {
            filter = logFile.useLogFilter(null);
            filter.setFirstKeyword(value);
            filter.setNumSurroundingLines(aroundString);
        } 
        filter.setFilterEnabled(true);
        
        IFilterListener listener = new IFilterListener() {
            @Override
            public void filterPerformed(FilterList filterList) {
                filter.setFilterEnabled(false);
            }
        };
        logFile.addFilterListener(listener);

        try {
            logFile.updateFilter();
        } catch (Exception e) {
            logFile.removeFilterListener(listener);
            filter.setFilterEnabled(false);
        }
    }

    @Override
    public void applyFilter(String resName, String filterString)
            throws Exception {
        applyFilter(resName, filterString, -1);
    }

    @Override
    public void saveFilter(String resName, String filterName,
            FilterResult filterResult) throws Exception {
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }

        Map<String, LogFilter> filters = logFile.getNamedFilterList();
        filterName = stripQuotes(filterName);
        LogFilter filter = filters.get(filterName);
        if (filter != null) {
            filters.remove(filter);
        }
        filter = new LogFilter(filterName, 0, 2);
        filters.put(filterName, filter);
        FilterItem item1 = null;
        int row = 0;

        for (FilterConditionRowResult condition : filterResult.getConditions()) {
            int col = 0;

            for (FilterConditionResult crow : condition.getRow()) {
                if (col > 1) { // since we created the filter with 2 cols
                    break;
                }

                item1 = new FilterItem();

                item1.setBoolConnector(crow.getOperator());
                item1.setKeyword(crow.getPattern());
                if (crow.isNot()!= null) item1.setNegateFlag(crow.isNot());
                if (crow.isRegEx()!= null) item1.setRegexFlag(crow.isRegEx());
                
                filter.setFilterItem(row, col, item1);
                ++col;
            }
            ++row;
        }
        if (!logFile.saveFilters()) { // store changes
            throw new Exception("Unable to save filters!");
        }
    }

    @Override
    public void deleteFilter(String resName, String filterName)
            throws Exception {
        LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
        if (logFile == null) {
            throw new Exception("Error when try to get an instance of LogFileAnalyser");
        }
        Map<String, LogFilter> filters = logFile.getNamedFilterList();
        filterName = stripQuotes(filterName);
        LogFilter filter = filters.get(filterName);
        if (filter != null) {
            filters.remove(filterName);
        }
        if (!logFile.saveFilters()) { // store changes
            throw new Exception("Unable to save filters!");
        }
    }

    @Override
    public URL fetchLogFileOrBundle(String hostName, String resName)
            throws Exception {
        DataInputStream in = null;
        BufferedReader br = null;
        if (resName != null && !resName.isEmpty()) {
            LogFileAnalyzer logFile = getLogFileAnalyzer(resName);
            if (logFile == null) {
                throw new Exception(
                        "Error when try to get an instance of LogFileAnalyser");
            }
            // need to save all files into a single object that can be returned
            // as the result of the call.
            String fName = File.separator + resName.replace(".logx", "")
                    + ".gz";
            String path = getWarBase() + fName;
            File destFile = new File(path);
            GZIPOutputStream out = null;
            out = new GZIPOutputStream(new FileOutputStream(destFile));
            List<RAFile> files = logFile.getAllFiles();
            for (RAFile f : files) {
                br = null;
                String fname = f.getTargetFilePath(true);
                in = new DataInputStream(new FileInputStream(fname));
                br = new BufferedReader(new InputStreamReader(in));
                String strLine;
                // Read File Line By Line
                while ((strLine = br.readLine()) != null) {
                    strLine += "\n";
                    out.write(strLine.getBytes());
                }

                if (br != null) {
                    br.close();
                }

            }

            if (out != null) {
                out.close();

            }

            return destFile.toURI().toURL();
        } else {
            String manifestName = getManifestFileName(hostName);
            File manifest = new File(manifestName);
            if (!manifest.exists()) {
                throw new Exception("Manifest for '" + hostName
                        + "' not found.\n");
            } else {
                in = new DataInputStream(new FileInputStream(manifestName));
                br = new BufferedReader(new InputStreamReader(in));
                String strLine;
                // Find bundle name
                while ((strLine = br.readLine()) != null) {
                    if (strLine.startsWith("bundle=")) {
                        int indx = strLine.lastIndexOf(File.separatorChar);
                        strLine = strLine.substring(indx);
                        return new File(strLine).toURI().toURL();
                    }
                }

                if (br != null) {
                    br.close();
                }

            }
        }
        return null;
    }

    @Override
    public void deleteAllLogsFromHost(String hostName) throws Exception {
        StringBuilder errorMsg = new StringBuilder();
        boolean failed = false;
        File f = null;
        BufferedReader br = null;
        DataInputStream in = null;
        String manifestName = getManifestFileName(hostName);
        File manifest = new File(manifestName);
        if (!manifest.exists()) {
            throw new Exception("Manifest for '" + hostName + "' not found.");
        }
        try {
            File project = prjManager.getDefaultProject();
            in = new DataInputStream(new FileInputStream(manifestName));
            br = new BufferedReader(new InputStreamReader(in));
            String strLine;
            String fname;
            //Read File Line By Line
            while ((strLine = br.readLine()) != null)   {
                if (strLine.startsWith("bundle=")) {
                    fname = strLine.substring(7);
                    f = new File(fname);
                    try {
                        f.delete();
                    } catch (Exception e) {
                        errorMsg.append("Failed to delete '" + fname + "'.\n");
                        failed = true;
                    }
                }
                if (!strLine.startsWith("file=")) {
                    continue;
                }
                String resName = stripQuotes(strLine.substring(5));
                LogFileAnalyzer logFile = prjManager.getLogFileAnalyzer(resName);
                if (logFile == null) {
                    continue;
                }
                // delete files saved in the cache
                fname = "";
                List<RAFile> files = logFile.getAllFiles();
                if (files != null) {
                    for (RAFile logx : files) {
                        try {
                            fname = logx.getTargetFilePath(true);
                            File fdel = new File(fname);
                            fdel.delete();
                        } catch (Exception e) {
                            failed = true;
                            errorMsg.append("Failed to delete '" + fname + "' " +
                                    e.getMessage() + "\n");
                        }
                    }
                }
                File file = new File(project, resName);
                if (file.exists()) {
                    try {
                        file.delete();
                    } catch (Exception e) {
                        failed = true;
                        errorMsg.append("Failed to delete '" + resName + "': " + e.getMessage() + "\n");
                    }
                }
                prjManager.remove(LogProjectManager.PROJECT_DEFAULT_NAME, resName);
            }
        } catch (Exception e) {
            errorMsg.append(e.getMessage() + "\n");
            failed = true;
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                    errorMsg.append(e.getMessage() + "\n");
                }
            }
            if (!failed) {
                // only delete the manifest if all other deletes went through, so that
                // if some of them failed, the user will have a chance to retry them.
                f = new File(manifestName);
                try {
                    f.delete();
                } catch (Exception e) {
                    errorMsg.append("Failed to delete '" + manifestName + "': " + e.getMessage() + "\n");
                }
            }
        }
        if (errorMsg.length() > 0) {
            throw new Exception ("Error deleting logs for '" +
                    hostName + "': " + errorMsg.toString());
        }
    }

    @Override
    public void update(Observable o, Object arg) {
        if (o instanceof LogRefreshRunnable) {
            LogRefreshRunnable task = (LogRefreshRunnable) o;
            String resName = (String) arg;
            ProgressResult progsResult = task.getProgressResult();
            if (progsResult.getStatus() == ProgressStatus.ERROR
                    || progsResult.getStatus() == ProgressStatus.DONE) {
                try {
                    // We are cloning to allow the threads be garbage collected
                    mapResnameProgress.put(resName, progsResult.clone());
                } catch (CloneNotSupportedException e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }
    }

    protected String getManifestFileName(String host) {
        String result = null;
        File project = prjManager.getDefaultProject();
        if (project != null) {
            result = project.getAbsolutePath() + File.separator + normalizeHostName(host)
                    + "_manifest.txt";
        }
        return result;
    }

    protected String getVMSupportFileName(String host) {
        String destLocation = getWarBase() + File.separator + normalizeHostName(host) + "_vmsupport.tgz";
        return destLocation;
    }

    public String getWarBase() {
        return publicFolder;
    }

    protected String stripQuotes(String s) {
        return s.replaceAll("\"", "").replaceAll("'", "");
    }

    protected LogFileAnalyzer getLogFileAnalyzer(String resName) throws Exception {
        LogFileAnalyzer logFile = prjManager.getLogFileAnalyzer(resName);
        logFile.open();

        return logFile;
    }

    protected LogProjectManager getLogProjectManager() {
        return prjManager;
    }

    private String normalizeHostName(String host) {
        String separator = "_";
        // Strip out '.' separator for IPv4 and ':' separator for IPv6
        return host.replaceAll("\\.", separator).replaceAll(":", separator);
    }
}
