/***********************************************************
 * Copyright 2011 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 ***********************************************************/

package com.vmware.vide.vlogbrowser.service;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.chart.IFilterListener;
import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.fileops.BaseLogWorker;
import com.vmware.vide.vlogbrowser.core.fileops.LogFileManager;
import com.vmware.vide.vlogbrowser.core.model.FilterList;
import com.vmware.vide.vlogbrowser.core.model.ILogItemList;
import com.vmware.vide.vlogbrowser.core.model.LogFilter;
import com.vmware.vide.vlogbrowser.core.model.LogItem;
import com.vmware.vide.vlogbrowser.core.parser.ConfigParser;
import com.vmware.vide.vlogbrowser.core.parser.DateMatcher;
import com.vmware.vide.vlogbrowser.core.parser.LogFormat;
import com.vmware.vide.vlogbrowser.core.parser.LogItemMatcher;
import com.vmware.vide.vlogbrowser.core.parser.ManifestFile;
import com.vmware.vide.vlogbrowser.core.parser.MultiMatcher;
import com.vmware.vide.vlogbrowser.core.parser.MultiMatcherFactory;
import com.vmware.vide.vlogbrowser.core.parser.RAFile;

/**
 * Collection of methods to analyze a log file.
 * This class must not use any UI elements, since
 * it is used by the Log Browser web provider.
 *
 * @author gcaprino
 */

public class LogFileAnalyzer {

    public String _name;
    public String _location;
    public transient List<IFilterListener> filterListListener;
    private FilterList filterList;

//    private ReentrantLock refreshLock = new ReentrantLock();

    private ConfigParser cfgParser;
    private LogFileManager logFileMgr;
    private LogFormat myLogFormat;
    private LogFilter fLogFilter;           // which LogFilter to use
    private LogFilter defaultLogFilter;     // default LogFilter to use with basic strings
    private Map<String, LogFilter> namedLogFilters;
    private BaseLogWorker logWorker = new BaseLogWorker();

    private static int MAX_FIND_COUNT = 1000;
    private DateFormat stdDateFormat;
    public static final String DATE_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss.SSS";
    
    private static final Logger logger = LoggerFactory.getLogger(LogFileAnalyzer.class);

    public LogFileAnalyzer(String name, String location) {
        _name = name;
        _location = location;
        // Create data containers
        filterList = new FilterList(name);
        defaultLogFilter = new LogFilter("", 0, LogBrowserConsts.DEFAULT_COLUMN_COUNT);
        fLogFilter = defaultLogFilter;
        filterListListener = new LinkedList<IFilterListener>();
        stdDateFormat = new SimpleDateFormat(DATE_FORMAT_STRING);
    }

//    public void reportReadError() {
//        /*
//        LogDialog dialog =
//            new LogDialog(viewer.getControl().getShell(),
//                    "Read Error",
//                    null,
//                    "Error reading file",
//                    MessageDialog.ERROR,
//                    new String[] { IDialogConstants.OK_LABEL }, 0);
//        dialog.setMessage("The input log file could not be read due to an error: " +
//                e.getMessage());
//        dialog.open();
//        */
//    }

    public boolean open() throws Exception {
        boolean parsed = false;
        if (cfgParser == null) {
            try {
                File cfgFile = new File(_location);
                InputStream configInputStream = new FileInputStream(cfgFile);
                cfgParser = new ConfigParser(cfgFile, configInputStream);
                logFileMgr = new LogFileManager(cfgParser);
                myLogFormat = cfgParser.getLogFormat();
                namedLogFilters = cfgParser.getLogFilters();
                
                // Get the staging directory, where the manifest file will be
//                File stagingDirectory = logFileMgr.getStagingDirectory();
//                File[] manifestFile = stagingDirectory.listFiles(new FileFilter(){
//                    @Override
//                    public boolean accept(File arg0) {
//                        return ManifestFile.MANIFEST_TEMPLATE.equals(arg0.getName());
//                    }
//                });
//                if (manifestFile != null && manifestFile.length > 0) {
//                    parsed = true;
//                }
                parsed = verifyIfParsedBefore();
                
                

                
            } catch (Exception e) {
                // re-throw exception to handle in editor
//                logWorker.parseFinished();
//                throw new Exception(e);
    //            throw new PartInitException(e.getMessage(), e);
                logger.error(e.getMessage(), e);
            }
            if (!parsed) {
                boolean forceReload = false;

                logger.debug("Waiting for access to log entry list...");
                try {
                    logger.debug("Initializing log entry table...");
                    // filterList.clear();
                    logger.debug("Checking log file staging area...");
                    myLogFormat.closeAllRandAccFiles();
                    List<RAFile> raFiles = null;
                    logFileMgr.prepareStagingDir(forceReload);
                    if (!forceReload) {
                        raFiles = logFileMgr.loadFilesFromManifest();
                    }
                    logger.debug("Preparing to load " + _name);

                    logWorker.setEventList(filterList);
                    logWorker.setLogFormat(myLogFormat);
                    logWorker.setLogWorkerType("FILE");
//                    logWorker.setRefreshLock(refreshLock);
                    logWorker.setLogFileMgr(logFileMgr);

                    if ((raFiles == null) || (raFiles.size() == 0)) {
                        raFiles = logFileMgr.getFileList();
                        if (raFiles.size() == 0) {
                            // 632386: call parser so it can send job completion
                            // event to ParserJobListener
                            logWorker.parseFilesIntoTable();
                            return false;
                        }
                        int lim = myLogFormat.getSizeLimitMB();
                        if (lim != -1) {
                            // it's going to be a large data set, so limit it to
                            // the latest entries
                            raFiles = logFileMgr.trimFileList(raFiles, lim);
                        }
                        logWorker.marshalLogFiles(raFiles);
                    }
                    /*
                     * final List<RAFile> raFiles2 = raFiles;
                     * Display.getDefault().syncExec(new Runnable() { public
                     * void run() { String msg =
                     * raFiles2.get(0).toShortDisplayString() +
                     * ((raFiles2.size() > 1) ? "..." : "");
                     * txtLogFilesLoaded.setText(msg); } });
                     */
                    // String logFilesInfoStr =
                    // LogFileUtils.getFileListStr(raFiles);
                    myLogFormat.loadNewRAFileList(raFiles);
                    logWorker.parseFilesIntoTable();
                } catch (Exception e) {
                    // reportReadError();
                    // logWorker.parseFinished();
                    logger.error(e.getMessage(), e);
                    return false;
                }
                parsed = true;
            } else {
                myLogFormat.loadNewRAFileList(getRAFiles());
            }
        }
        /*
        LogLoadRunner llRunner = new LogLoadRunner(myLogTableViewer, false, "Loading log files.");
        try {
            // new ProgressMonitorDialog(getSite().getShell()).run(false, false, llRunner);
            llRunner.run(null);
        } catch (Exception e1) {
            e1.printStackTrace();
            Activator.log(e1);
        }
        */
        return true;

    }
    
    private boolean verifyIfParsedBefore() {
        return false;
    }

    private List<RAFile> getRAFiles() {
        List<RAFile> result = null;
        try {
            result = logFileMgr.loadFilesFromManifest();
            if (result != null) {
                for (RAFile raFile: result) {
                    raFile.open();
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return result;
    }

    public boolean save() {
        String content = cfgParser.getXmlFromDocument();
        File cfgFile = new File(_location);
        FileWriter writer = null;
        try {
            writer = new FileWriter(cfgFile);
            writer.write(content);
            return true;
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
            return false;
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    logger.error(e.getMessage(), e);
                    return false;
                }
            }
        }
    }

    public boolean saveFilters() {
        Set<Entry<String, LogFilter>> names = namedLogFilters.entrySet();
        Iterator<Entry<String, LogFilter>> x = names.iterator();
        while (x.hasNext()) {
            Entry<String, LogFilter> e = x.next();
            try {
                cfgParser.addLogFilterNode(e.getValue()); // will add or replace
            } catch (Exception e1) {
                logger.error(e1.getMessage(), e1);
                return false;
            }
        }
        return save();
    }

    public synchronized void addFilterListener(IFilterListener listener) {
        filterListListener.add(listener);
    }

    public synchronized void removeFilterListener(IFilterListener listener) {
        filterListListener.remove(listener);
    }

    public void notifyFilteredListeners() {
        for (IFilterListener listener : filterListListener) {
            listener.filterPerformed(filterList);
        }
    }

    public FilterList getFilterList() {
        return filterList;
    }

    public LogFormat getLogFormat() {
        return myLogFormat;
    }

    public boolean isWorking() {
        return logWorker.isWorking();
    }

    public LogFilter useLogFilter(String name) {
        if (name == null || name.isEmpty()) {
            fLogFilter = defaultLogFilter;
        } else {
            Map<String, LogFilter> filters = getNamedFilterList();
            fLogFilter = filters.get(name);
            if (fLogFilter == null) {
                fLogFilter = defaultLogFilter;
            }
        }
        return fLogFilter;
    }

    public LogFilter getLogFilter() {
        return fLogFilter;
    }
/*
    public void setTxtFilterStr(String value) {
        // TODO Auto-generated method stub

    }
*/
    public void updateFilter() throws Exception {
        //long currentSelectionIndex = getCurrentIndexNum();
        //fLogFilter = logFilter;

        if (!fLogFilter.isFilterEnabled()) {
            logger.debug("Removing filter from table...");
            filterList.setMatcher(null);

        // mark all items as unfiltered
        for (int i = 0; i < filterList.size(); i++) {
            LogItem item = filterList.get(i);
            item.setAlwaysMatched(false);
            item.setGroupId(0);
            filterList.updateLogItem(i, item);
        }

        }else{
        	 logger.debug("Removing previous filter from table...");
            // get the best matcher based on the filter configured
            MultiMatcher matcher = MultiMatcherFactory.getMultiMatcher(fLogFilter, getLogFormat());

            // apply new filter to the table
            int threshold = fLogFilter.getNumSurroundingLines();
            if (threshold > 0) {
                logger.debug("Building filter... Total log entries to process: " + filterList.size());
                logger.debug("Processing log entries...", filterList.size());
                ILogItemList items = filterList.getUnfilteredItems();
                for (int i = 0; i < items.size(); i++) {
                    LogItem myItem = items.get(i);
                    if (matcher.matchesSomeField(myItem)) {
                        myItem.setAlwaysMatched(true);
                        items.updateLogItem(i, myItem);
                        for (int count = 1; count <= threshold; count++) {
                            int arrayIndex;
                            if ((arrayIndex = (i - count)) >= 0) {
                                LogItem aLogItem = items.get(arrayIndex);
                                aLogItem.setAlwaysMatched(true);
                                items.updateLogItem(arrayIndex, aLogItem);
                            }
                            if ((arrayIndex = i + count) < items.size()) {
                                LogItem aLogItem = items.get(arrayIndex);
                                aLogItem.setAlwaysMatched(true);
                                items.updateLogItem(arrayIndex, aLogItem);
                            }
                        }
                    }
                }

                /*
                 * Now compute for each item which group it belongs to.
                 * A group is defined as a sequence of contiguous items.
                 */
                logger.debug("Alternating background color for adjacent match groups...");
                int groupId = 0;
                for (int i = 0; i < items.size(); i++) {
                    LogItem item = items.get(i);
                    if (!item.isAlwaysMatched()) {
                        continue;
                    }
                    do {
                        item.setGroupId(groupId);
                        items.updateLogItem(i, item);
                    } while (i+1 < items.size() &&
                            (item = items.get(++i)) != null && item.isAlwaysMatched());

                    ++groupId;
                }
            } 

            logger.debug("Applying new filter to table...");
            filterList.setMatcher(matcher);
        }

        notifyFilteredListeners();
        //gotoIndexNum(currentSelectionIndex);
    }

    public Map<String, LogFilter> getNamedFilterList() {
        return namedLogFilters;
    }

    //
    //
    //

    public List<Integer> findPattern(String findKeyword, int startingRow) {
        FilterList filterList;
        ILogItemList items;
//        String fltr = "";// filterString; //viewer.getTxtFilterStr();
        filterList = getFilterList();
//        if (fltr == null || fltr.isEmpty()) {
//            items = filterList.getUnfilteredItems();
//        } else {
        	items = filterList.getFilteredItems();
//        }
        LogItemMatcher lim = new LogItemMatcher(findKeyword, myLogFormat);
        List<Integer> matchedIndices = new ArrayList<Integer>();
        for (int i = startingRow; i < filterList.size(); i++) {
            LogItem li = items.get(i);
            if (lim.matchesSomeField(li)) {
                matchedIndices.add(i);
                if (matchedIndices.size() == MAX_FIND_COUNT) {
                    break;
                }
            }
        }
        return matchedIndices;
    }

    public int findClosestDate(Date findDate) {
        long findDateInMillis = findDate.getTime() - myLogFormat.getDateDelta();
        DateMatcher dateMatcher = new DateMatcher(new Date(findDateInMillis));
        // This is a sequential search, i.e. O(n) rather than tree-based, O(log n), for 2 reasons:
        // 1. There is no guarantee that the current filter list structure of log entries
        //  is currently sorted by date (since the user can sort by any column).
        // 2. For whatever reason, log entries are sometimes written to log files out-of-timestamp-order.
        for (int i = 0; i < filterList.size(); i++) {
            LogItem logItem = filterList.get(i);
            dateMatcher.submitMatchCandidate(logItem, i);
        }
        int result = dateMatcher.getClosestMatchIndex();
        return result;
    }

    public List<Integer> findDate(String theDate) {
        List<Integer> foundRow = new ArrayList<Integer>();
        try {
            Date date = stdDateFormat.parse(theDate);
            int row = findClosestDate(date);
            foundRow.add(row);
        } catch (ParseException e) {
            // should be impossible since date is generated by UI code, not by user
            e.printStackTrace();
        }
        return foundRow;
    }

    public List<RAFile> getAllFiles() {
        if (myLogFormat == null) {
            return null;
        }
        return myLogFormat.getRandAccFileList();
    }
}
