package com.vmware.vide.vlogbrowser.service;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.Observable;

import org.apache.http.HttpEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vide.vlogbrowser.core.consts.LogBrowserConsts;
import com.vmware.vide.vlogbrowser.core.fileops.LogFileManager;
import com.vmware.vide.vlogbrowser.core.fileops.LogxFileSetCreator;
import com.vmware.vide.vlogbrowser.core.model.LogxFile;
import com.vmware.vide.vlogbrowser.service.model.ProgressResult;
import com.vmware.vide.vlogbrowser.service.model.ProgressResult.ProgressStatus;

public class LogRefreshRunnable extends Observable implements Runnable {

    private ProgressResult progressResult = new ProgressResult();
    private StringBuilder errorMsg = new StringBuilder();

    private HttpEntity entity;
    private String destLocation;
    private String manifestName;
    private String resName;

    private static final Logger logger = LoggerFactory.getLogger(LogRefreshRunnable.class);

    private static final int BUFFER_SIZE = 4 * 1024; // 4KB

    public LogRefreshRunnable(HttpEntity entity, String destLocation, String manifestName,
            String host, String resName) {
        this.entity = entity;
        this.destLocation = destLocation;
        this.manifestName = manifestName;
        this.resName = resName;
        this.progressResult.getLogFileResult().setName(host);
    }
    
    private void setStatus(ProgressStatus status) {
        progressResult.setStatus(status);
        setChanged();
        notifyObservers(resName);
    }

    @Override
    public void run() {
        setStatus(ProgressStatus.BUSY);
        BufferedInputStream bis = null;
        OutputStream outstream = null;
        BufferedWriter out = null;
        try {
            InputStream instream = entity.getContent();
            bis = new BufferedInputStream(instream);
            outstream = new BufferedOutputStream(new FileOutputStream(destLocation), BUFFER_SIZE);
            int cnt = 0;
            int off = 0;

            byte[] cbuf = new byte[BUFFER_SIZE];
            progressResult.setTotalWork(entity.getContentLength());
            logger.debug("Starting downloading ("+resName+")");
            while ((cnt = bis.read(cbuf)) > 0) {
                outstream.write(cbuf, 0, cnt);
                off += cnt;
                progressResult.incrementWork(cnt);
            }
            outstream.flush();
            outstream.close();
            logger.debug("Finishing downloading ("+resName+")");

            // create the .logx files related to this tgz file
            LogxFileSetCreator logCreator = new LogxFileSetCreator();
            LogxFile selectedFormat = null;
            String tgzFilepath = destLocation;
            String parentDir = destLocation.substring(0,
                    destLocation.lastIndexOf(File.separatorChar) + 1);
            String filePatterns = destLocation.substring(destLocation
                    .lastIndexOf(File.separatorChar) + 1);

            String prjName = LogProjectManager.PROJECT_DEFAULT_NAME;
            logCreator.setProject(prjName);
            logCreator.addLogxFilesToList(LogFileManager.LOCALHOST, selectedFormat,
                    LogBrowserConsts.LOG_FORMAT_VMSUPPORT_ARCHIVE, tgzFilepath, parentDir,
                    filePatterns);

            // create the manifest related to this tgz file

            Calendar now = Calendar.getInstance();
            String nowString = "" + now.get(Calendar.YEAR) + "-" + (now.get(Calendar.MONTH) + 1)
                    + "-" + now.get(Calendar.DAY_OF_MONTH) + " " + now.get(Calendar.HOUR_OF_DAY)
                    + ":" + now.get(Calendar.MINUTE) + ":" + now.get(Calendar.SECOND);
            FileWriter writer = new FileWriter(manifestName);
            out = new BufferedWriter(writer);
            out.write("bundle=");
            out.write(tgzFilepath);
            out.newLine();
            out.write("source=");
            out.write(resName);
            out.newLine();
            for (LogxFile f : logCreator.getLxFiles()) {

                try {
                    LogxFileSetCreator.createLogxFile(prjName, f);
                } catch (Exception e1) {
                    errorMsg.append("Error creating '");
                    errorMsg.append(f.getLogxFileName());
                    errorMsg.append("' : ");
                    errorMsg.append(e1.getMessage());
                    errorMsg.append("\n");
                    continue;
                }

                out.write("file=");
                out.write(f.getLogxFileName());
                out.write("\n");
                out.write("type=");
                out.write(f.getLogFormatName());
                out.write("\n");
                out.write("date=");
                out.write(nowString);
                out.write("\n");
            }
            progressResult.getLogFileResult().setDate(nowString);
            setStatus(ProgressStatus.DONE);
        } catch (Exception e) {
            setStatus(ProgressStatus.ERROR);
            errorMsg.append("Error: ");
            errorMsg.append(e.getMessage());
            errorMsg.append("\n");
        } finally {
            try {
                if (bis != null) {
                    bis.close();
                }
                if (out != null) {
                    out.close();
                }
                if (outstream != null) {
                    outstream.close();
                }
            } catch (IOException e) {
                setStatus(ProgressStatus.ERROR);
                errorMsg.append(e.getMessage());
            }
        }
    }

    public ProgressResult getProgressResult() {
        return progressResult;
    }

    public StringBuilder getErrorMsg() {
        return errorMsg;
    }

}
