package com.vmware.samples.chassisservice;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.vmware.samples.chassisservice.model.Chassis;
import com.vmware.samples.chassisservice.model.ChassisInfo;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Simplified data store for the Chassis objects, and related utilities.
 *
 * ***************************************************************************** *
 * IMPORTANT: this implementation uses in-memory data to keep the setup easy,    *
 * a real implementation should retrieve data from a remote server or database   *
 * and have very minimal or no storing/caching in the java layer because the     *
 * server must remain stateless and be able to scale out.                        *
 * ***************************************************************************** *
 *
 * Note that this class is thread-safe but doesn't deal with complex operations
 * or large data sets. It is not intended to be used as-is!
 */
public class ObjectStore {

   // We start with 4 chassis but more can be created in the UI
   private static final int CHASSIS_INITIAL_COUNT = 4;

   // The custom object type
   private static final String CHASSIS_TYPE = ChassisDataAdapter.CHASSIS_TYPE;

   // Map of chassis objects used in the sample.
   // The key is the object's uid and the value the Chassis instance.
   private final Map<String, Chassis> _objectMap =
         new HashMap<String, Chassis>(CHASSIS_INITIAL_COUNT);

   // Internal index used to create unique ids
   private static int _index = 0;

   private final VimObjectReferenceService _objectRefService;

   public ObjectStore(VimObjectReferenceService objectReferenceService) {
      _objectRefService = objectReferenceService;
   }

   /**
    * Initializes the local object store with 4 Chassis.
    * This bean init method is defined in bundle-context.xml.
    *
    * Note: initialization done during the bean creation must not be time-consuming.
    * A real implementation will need some back-end information that could be stored
    * in a properties file, or fetched from the plugin's vCenter extension,
    * or configured in the UI by a user.
    */
   public void init() {
      // Create the initial set of chassis
      for (int i = 0; i < CHASSIS_INITIAL_COUNT; i++) {
          String[] data = {"Chassis "+(i+1), "Server_Type "+ (i%3), "20in x 30in x 17in"};
          createChassis(new ChassisInfo(data), false);
      }
   }

   /**
    * Bean destroy method defined in bundle-context.xml.
    */
   public void destroy() {
      // nothing to clean-up in this sample.
   }

   /**
    * Get the current chassis objects in a thread-safe manner which allows the caller
    * to iterate through the list. Making a copy and exposing ObjectStore internals is
    * OK for this small example. A real implementation should provide a thread-safe
    * way of iterating through objects stored in a back-end server, without caching
    * them in memory.
    *
    * @return a copy of the current map of objects.
    */
   Map<String, Chassis> getObjects() {
      synchronized(_objectMap) {
         return new HashMap<String, Chassis>(_objectMap) ;
      }
   }

   /**
    * Access a Chassis from the internal object store.
    *
    * @param uid  Unique identifier.
    * @return the Chassis object for the given uid, or null.
    */
   Chassis getChassis(String uid) {
      synchronized(_objectMap) {
         return _objectMap.get(uid);
      }
   }

   /**
    * Remove a Chassis from the internal object store.
    *
    * @param uid  Unique identifier.
    * @return the Chassis object that was removed, or null if no object was found for that uid.
    */
   Chassis removeChassis(String uid) {
      synchronized(_objectMap) {
         return _objectMap.remove(uid);
      }
   }

   /**
    * Add a new Chassis to the object store
    *
    * @param chassisInfo
    *          The data used to create the chassis.
    * @param checkName
    *          true if the name should be checked first, to ensure unique names.
    * @return
    *          The new Chassis object, or null if the name is already taken.
    */
   Chassis createChassis(ChassisInfo chassisInfo, boolean checkName) {
      synchronized(_objectMap) {
         // Enforce unique names
         if (checkName && mapContainsName(_objectMap, chassisInfo.name)) {
            return null;
         }

         // Creates a unique object reference with an internal value chassis-N
         Object chassisReference = _objectRefService.getReference(
                  CHASSIS_TYPE, "chassis-"+(_index++), null);

         Chassis chassis = new Chassis(chassisInfo, chassisReference);
         String uid = _objectRefService.getUid(chassisReference);

         // Add chassis to object store
         _objectMap.put(uid, chassis);

         return chassis;
      }
   }

   /**
    * Helper method to enforce name uniqueness.
    *
    * @param map  The object map passed in a thread-safe manner.
    * @param name The name to check
    * @return true if the given chassis name is already taken.
    */
   private static boolean mapContainsName(Map<String, Chassis> map, String name) {
      Iterator<String> i = map.keySet().iterator();
      while(i.hasNext()) {
         String uid = i.next();
         Chassis chassis = map.get(uid);
         if (name.equals(chassis.getName())) {
            return true;
         }
      }
      return false;
   }

   /**
    * Update an existing chassis with new data.
    *
    * @param uid     The chassis unique identifier.
    * @param newInfo The new data.
    * @param chassisRef The chassis reference
    * @return true if the updated succeeded, false if the chassis didn't exist.
    */
   boolean updateChassis(String uid, ChassisInfo newInfo, Object chassisRef) {
      synchronized(_objectMap) {
         Chassis newChassis = new Chassis(newInfo, chassisRef);
         Chassis oldChassis = _objectMap.put(uid, newChassis);
         return (oldChassis != null);
      }
   }

}
