package com.vmware.samples.chassisui.views.wizardPages {

import com.vmware.data.Constraint;
import com.vmware.data.query.Comparator;
import com.vmware.data.query.events.DataByConstraintRequest;
import com.vmware.data.query.events.DataByQuerySpecRequest;
import com.vmware.data.query.events.MultiObjectDataResponse;
import com.vmware.data.query.QuerySpec;
import com.vmware.samples.chassisui.Util;
import com.vmware.samples.chassisui.model.ChassisInfo;
import com.vmware.samples.chassisui.model.ChassisSummaryDetails;
import com.vmware.widgets.controls.wizardclasses.PageInfo;
import com.vmware.widgets.controls.wizardclasses.PageState;
import com.vmware.widgets.controls.wizardclasses.WizardPageMediator;
import com.vmware.widgets.util.Dialogs;

import mx.controls.Alert;
import mx.collections.ArrayCollection;

[Event(name="{com.vmware.data.query.events.DataByQuerySpecRequest.REQUEST_ID}",
        type="com.vmware.data.query.events.DataByQuerySpecRequest")]
[Event(name="{com.vmware.data.query.events.DataByConstraintRequest.REQUEST_ID}",
        type="com.vmware.data.query.events.DataByConstraintRequest")]
/**
 * The mediator for the NamePage view. It extends the WizardPageMediator which gives it
 * the needed functions to easily integrate with the ChassisWizard.
 *
 */
public class NamePageMediator extends WizardPageMediator {

   // The view that this mediator controls.
   private var _view:NamePage;
   private var _pageInfo:PageInfo;

   // Internal variables
   private var _callback:Function;
   private var _callbackContext:Object;

   [InjectableView]
   /**
    * The view associated with this mediator.
    */
   public function set view(value:NamePage):void {
      _view = value;
   }

   /** @private */
   public function get view():NamePage {
      return _view;
   }

   /**
    * Data model inherited from the wizard.
    */
   private var _data:ChassisInfo;

   [Inherited]
   public function set data(value:ChassisInfo):void {
      if (_data != value) {
         _data = value;
      }
      if (_view != null && value != null) {
         // initialize text fields
         _view.chassisInfo = value;
      }
   }

   /** @private */
   public function get data():ChassisInfo {
      return _data;
   }

   override public function activate():void {
      if (this.pageInfo.initialized) {
         return;
      }
      if ((data != null) && (data.name != null) && (data.name.length > 0)) {
         this.pageInfo.initialized = true;
         return;
      }
      // We want to replace the initial empty name with a default value "Chassis (N+1)"
      // where N is the current number of chassis. A quick server request is done
      // using a Constraint to get the "name" property for all chassis objects,
      // but maxResultCount is set to 0 since we are not interested in the values!
      var qs:QuerySpec = Util.createSimpleConstraintQuerySpec("samples:Chassis", ["name"]);
      qs.resultSpec.maxResultCount = 0;
      dispatchEvent(DataByQuerySpecRequest.newInstance(qs));

      // Note that if the page initialization takes a few seconds you should bring up
      // a busy signal or progress bar to let users know.
   }

   [EventHandler(name=
         "{com.vmware.data.query.events.DataByQuerySpecRequest.RESPONSE_ID}")]
   /**
    * Process the result of the DataByQuerySpecRequest sent above.
    * We use this format instead of the normal ResponseHandler in order to access
    * the MultiObjectDataResponse object and its totalMatchedObjectCount field.
    */
   public function onDataByQuerySpecRetrieved(response:MultiObjectDataResponse):void {
      if (response.error != null) {
         Alert.show("Error: " + response.error.message);
         return;
      }
      // Number of chassis returned by the java ChassisDataAdapter
      var count:int = response.totalMatchedObjectCount;

      this.data.name = "Chassis " + (count+1);

      this.pageInfo.initialized = true;
   }

   override public function deactivate():void {
      // This can be used to do further cleanup when the view is de-activated
   }

   /**
    * Override the pageInfo method to implement the commit function that
    * is called whenever the "Next" button is clicked.
    */
   override public function set pageInfo(value:PageInfo):void {
      super.pageInfo = value;

      if (pageInfo != null) {
         pageInfo.commitFunction = onPageCommit;
      }
   }

   /**
    * Commit function called when "Next" is clicked.
    * Performs validation of user input and allows to move to the next page.
    *
    * Here we verify 2 things:
    * 1) client-side check that both fields are non empty
    * 2) if #1 passes, server-side check that the chassis name is not already taken
    *
    * Since #2 is asynchronous the validation callback must be made in the ResponseHandler.
    */
   private function onPageCommit(
         pageInfo:PageInfo,
         callback:Function,
         callbackContext:Object):void {

      _data.name = _view.chassisName.text;
      _data.serverType = _view.serverType.text;
      var emptyName:Boolean = (_data.name.length == 0);
      var emptyType:Boolean = (_data.serverType.length == 0);

      // Mark invalid fields with red border
      _view.chassisName.setStyle("borderColor", (emptyName ? "red" : "black"));
      _view.serverType.setStyle("borderColor", (emptyType ? "red" : "black"));

      if (emptyName || emptyType) {
         // No need to go further, the page is not valid and we don't let user click Next.
         changeState(PageState.INVALID);
         if (callback != null) {
            callback(pageInfo, false, callbackContext);
         }
         return;
      }

      // Additional server-side call to check if the name already exists,
      // i.e. we request all chassis objects whose name is equal to _data.name
      // (the ChassisSummaryDetails model is used for convenience as it includes "name")
      //
      // Note: a more advanced validation could be done by calling a java service, etc.

      var chassisNameConstraint:Constraint = Util.createPropertyConstraint(
           "samples:Chassis", "name", Comparator.EQUALS, _data.name);
      var request:DataByConstraintRequest =
           DataByConstraintRequest.newInstance(chassisNameConstraint, ChassisSummaryDetails);
      _callback = callback;
      _callbackContext = callbackContext;
      dispatchEvent(request);
   }

   /**
    * Handles the response to DataByConstraintRequest above. The method name
    * is not important, only its signature and the ResponseHandler tag are.
    * Note that the result is provided as an array of ChassisSummaryDetails items since
    * the request was using that model.
    **/
   [ResponseHandler(name=
         "{com.vmware.data.query.events.DataByConstraintRequest.RESPONSE_ID}")]
   public function onDataByConstraintRetrieved(request:DataByConstraintRequest,
                                   result:ArrayCollection, error:Error):void {
      if (error != null) {
         Alert.show("Add Chassis error: " + error.message);
         return;
      }

      // Page is valid if no chassis was found with the same name.
      var pageValid:Boolean = (result != null && result.length == 0);
      changeState(pageValid ? PageState.COMPLETE : PageState.INVALID);

      if (!pageValid) {
         Dialogs.showWarning(Util.getString("createAction.warning", [data.name]),
            Util.getString("chassis.createAction"));
      }
      // Callback must be made to allow (or not) the user to move to the next page
      if (_callback != null) {
         _callback(pageInfo, pageValid, _callbackContext);
      }
   }

}
}