package com.vmware.samples.customprovider;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.gson.Gson;

/**
 * Web Service example using Json data.
 *
 * This service uses the Google Elevation API and retrieves the elevation
 * of a location, given its latitude and longitude.
 */
public class GoogleService {

   private static final Log _logger = LogFactory.getLog(GoogleService.class);
   /**
    * The following are the set of constants, which are used for displaying and
    * retrieving the elevation information.
    */
   private static final String GOOGLE_ELEVATION_URL_PREFIX =
         "http://maps.googleapis.com/maps/api/elevation/json?locations=";
   public static final String ELEVATION = "Elevation";
   public static final String LATITUDE = "Latitude";
   public static final String LONGITUDE = "Longitude";

   /**
    * @param latitude
    * @param longitude
    * @return the ElevationData of a location using its latitude and longitude
    * @throws IOException
     */
   public ElevationData getElevationData(String latitude, String longitude)
                                                         throws Exception {

      // Build the URL for the Google Maps Elevation API.
      // See http://code.google.com/apis/maps/documentation/elevation/
      String google_service_url = String.format("%s%s,%s&sensor=false",
            GOOGLE_ELEVATION_URL_PREFIX, latitude, longitude);

      URL source = null;
      try {
         source = new URL(google_service_url);
      } catch (MalformedURLException ex) {
         _logger.error(ex);
         throw ex;
      }

      // Create the connection object, using a proxy if there is one defined in
      // System properties (add these VM args in the Virgo launch configuration:
      //  -Dhttp.proxyHost="your.proxy.host" -Dhttp.proxyPort=YourPortNumber
      URLConnection urlConnection = null;
      try {
         urlConnection = source.openConnection();
      } catch (Exception ex) {
         _logger.error("Error opening connection: " + ex);
         throw ex;
      }

      // An input stream that reads from this open connection.
      InputStreamReader streamReader = null;
      ApiResult apiResult = null;
      ElevationData data = null;

      try {
         streamReader = new InputStreamReader(urlConnection.getInputStream());
         BufferedReader br = new BufferedReader(streamReader);

         // Create a Java Object from the JSON object.
         Gson gson = new Gson();
         apiResult = gson.fromJson(br, ApiResult.class);

       } catch (Exception ex) {
          String errorMsg = "Error getting data from " + urlConnection +
                ": " + ex.getMessage() + ". \n\n(A timeout usually means that" +
                " you need to change your Proxy settings for this sample.)";
          _logger.error(errorMsg);
          throw new Exception(errorMsg);
       } finally {
          if (streamReader != null) {
             try {
                streamReader.close();
             } catch (IOException ex) {
                _logger.error("Could not close the stream : " + ex);
             }
          }
       }

       if (apiResult != null) {
          if (apiResult.status.equals("OK")) {
             data = new ElevationData();
             ElevationResult elevResult = apiResult.results.get(0);
             data.elevation = Double.toString(elevResult.elevation);
             data.latitude = Double.toString(elevResult.location.lat);
             data.longitude = Double.toString(elevResult.location.lng);

             int commaIndex = data.elevation.indexOf('.');
             if (commaIndex > 0 && commaIndex < (data.elevation.length() - 2)) {
                data.elevation = data.elevation.substring(0, commaIndex + 2);
             }
          } else {
             String errorMsg = "Error retrieving elevation data at " +
                   google_service_url + ": " + apiResult.status;
             _logger.error(errorMsg);
             throw new Exception(errorMsg);
          }
       }
       return data;
   }

   /**
    * Result of the Goodle Map API for elevation data.
    * The Java class structure and field names must match the Json fields.
    * See http://code.google.com/apis/maps/documentation/elevation/
    */
   private class ApiResult {
      @SuppressWarnings("unused")
      ApiResult() {};
      public List<ElevationResult> results;
      public String status;
   }

   private class ElevationResult {
      @SuppressWarnings("unused")
      ElevationResult() {};
      public Double elevation;
      public Location location;
   }

   private class Location {
      @SuppressWarnings("unused")
      Location() {};
      public Double lat;
      public Double lng;
   }
}
