package com.vmware.samples.viewspropertiesui {
import com.vmware.data.query.*;
import com.vmware.core.model.IResourceReference;
import com.vmware.data.*;

/**
 * Helper functions to create data query Constraints.
 */
public class QuerySpecUtil {

   /**
    * Creates a Constraint to match related objects to a given object.
    *
    * @param object
    *   The source object of the relationship.
    *
    * @param relationship
    *   Relationship name from the source object to the target objects to match.
    *
    * @param targetType
    *   Type of objects to retrieve.
    *   If not specified, it is implicitly figured out from the given relationship.
    *
    * @return The new RelationalConstraint.
    *
    * @example createConstraintForRelationship(hostRef, "vm") is a contraint
    *   to return all VMs of a host.
    */
   public static function createConstraintForRelationship(
         object:IResourceReference,
         relationship:String,
         targetType:String=null):Constraint {

      // match the given object with an ObjectIdentityConstraint
      var objectConstraint:ObjectIdentityConstraint =
            createObjectIdentityConstraint(object);

      // create a relational constraint for the given relationship to the source object
      var relationalConstraint:RelationalConstraint = createRelationalConstraint
            (relationship, objectConstraint, targetType, true);

      return relationalConstraint;
   }

   /**
    * Creates a Constraint to match an object.
    *
    * @param objectRef  Reference to the object.
    *
    * @return The new ObjectIdentityConstraint.
    */
   public static function createObjectIdentityConstraint(
            objectRef:IResourceReference):ObjectIdentityConstraint {
      var constraint:ObjectIdentityConstraint = new ObjectIdentityConstraint();
      constraint.target = objectRef;
      constraint.targetType = objectRef.type;

      return constraint;
   }

   /**
    * Creates a Constraint based on some relation to given objects.
    *
    * @param relationship
    *       The relation to match (the source type of the relation
    *       is defined by the isSourceConstrainted property)
    *
    * @param constraintOnRelatedEntity
    *       Constraint on the objects related to the targeted objects by the
    *       given relationship.
    *
    * @param targetType
    *       Type of the objects to retrieve
    *
    * @param isSourceConstrained
    *       Indicates whether the constraintOnRelatedEntity applies to the source,
    *       as opposed to the target of the relationship given in this instance.
    *       (false by default)
    *
    * @return The new RelationalConstraint.
    */
   public static function createRelationalConstraint(
            relationship:String,
            constraintOnRelatedEntity:Constraint,
            targetType:String,
            isSourceConstrained:Boolean = false):RelationalConstraint {

      var constraint:RelationalConstraint = new RelationalConstraint();
      constraint.relation = relationship;
      constraint.constraintOnRelatedObject = constraintOnRelatedEntity;
      constraint.targetType = targetType;
      constraint.hasInverseRelation = isSourceConstrained;
      return constraint;
   }

   /**
    * Creates a Constraint based on an object's property.
    *
    * @param objectType
    *       Type of object to retrieve.
    * @param propertyName
    *       Name of the property to be matched.
    * @param comparator
    *       The operator to use for comparison.
    *       ('equals', 'unequals', 'matches', 'ranges_between', etc.).
    * @param comparableValue
    *       The value to compare.
    *
    * @return The new PropertyConstraint.
    */
   public static function createPropertyConstraint(
         objectType:String,
         propertyName:String,
         comparator:Comparator,
         comparableValue:Object):PropertyConstraint {

      var constraint:PropertyConstraint = new PropertyConstraint();
      constraint.targetType = objectType;
      constraint.propertyName = propertyName;
      constraint.comparator = comparator;
      constraint.comparableValue = comparableValue;

      return constraint;
   }

   /**
    * Creates a Constraint combining several constraints
    *
    * @param nestedConstraints
    *       Array of constraints to add into the composite constraint.
    *
    * @param conjoiner
    *       "AND" or "OR" operator to combine the nested constraints.
    *
    * @param defaultTargetType
    *       The default target type to use if the target types of the individual
    *       constraints do not match.
    *
    * @return  The new CompositeConstraint.
    */
   public static function createCompositeConstraint(
            nestedConstraints:Array,
            conjoiner:Conjoiner,
            defaultTargetType:String):CompositeConstraint {

      // create the result constraint and initialize it
      var compositeConstraint:CompositeConstraint = new CompositeConstraint();
      compositeConstraint.nestedConstraints = nestedConstraints;
      compositeConstraint.conjoiner = conjoiner;

      // determine the target type:
      // - if all nested constraints have the same type, use it.
      // - otherwise, use the default target type.
      compositeConstraint.targetType = defaultTargetType;
      for (var i:int = 0; i < nestedConstraints.length; i++) {
         // no comparison needed for the first one, just set it as our target
         if (i == 0) {
            compositeConstraint.targetType = Constraint(nestedConstraints[i]).targetType;
            continue;
         }

         // check for the same target type
         if (compositeConstraint.targetType != Constraint(nestedConstraints[i]).targetType) {
            // heterogeneous set of target types detected in nested constraints
            compositeConstraint.targetType = defaultTargetType;
            break;
         }
      }

      return compositeConstraint;
   }

} // end class
} // end package
