package com.vmware.samples.wssdkprovider;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.vmware.vim25.DynamicProperty;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.ObjectContent;
import com.vmware.vim25.ObjectSpec;
import com.vmware.vim25.PropertyFilterSpec;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.SelectionSpec;
import com.vmware.vim25.ServiceContent;
import com.vmware.vim25.TraversalSpec;
import com.vmware.vim25.VimPortType;
import com.vmware.vim25.VimService;
import com.vmware.vim25.VirtualDevice;
import com.vmware.vim25.VirtualDisk;
import com.vmware.vim25.VirtualMachineConfigInfo;
import com.vmware.vise.data.query.DataServiceExtensionRegistry;
import com.vmware.vise.data.query.PropertyProviderAdapter;
import com.vmware.vise.data.query.PropertyRequestSpec;
import com.vmware.vise.data.query.PropertyValue;
import com.vmware.vise.data.query.ResultItem;
import com.vmware.vise.data.query.ResultSet;
import com.vmware.vise.data.query.TypeInfo;
import com.vmware.vise.security.ClientSessionEndListener;
import com.vmware.vise.usersession.ServerInfo;
import com.vmware.vise.usersession.UserSession;
import com.vmware.vise.usersession.UserSessionService;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Example of a property provider using the vSphere Web Services SDK (vim25)
 */
public class VmDataProviderImpl implements
      PropertyProviderAdapter, ClientSessionEndListener {

   private static final Log _logger = LogFactory.getLog(VmDataProviderImpl.class);
   private static final String VM_TYPE = "VirtualMachine";
   /*
    * Custom VM property provided by this adapter. It must be qualified with its
    * own namespace to avoid collisions with the vSphere default space
    */
   private static final String VMDATA_PROPERTY = "samples:vmData";

   private final UserSessionService _userSessionService;
   private final VimObjectReferenceService _vimObjectReferenceService;

   private static final String SERVICE_INSTANCE = "ServiceInstance";

   /** object for access to all of the methods defined in the vSphere API */
   private static VimPortType _vimPort = initializeVimPort();

   private static VimPortType initializeVimPort() {
      // Static initialization is preferred because it takes a few seconds.
      VimService vimService = new VimService();
      return vimService.getVimPort();
   }

   /**
    * Static Initialization block, which will make this client trust all
    * certificates. USE THIS ONLY FOR TESTING.
    */
   static {
      HostnameVerifier hostNameVerifier = new HostnameVerifier() {
         @Override
         public boolean verify(String urlHostName, SSLSession session) {
            return true;
         }
      };
      HttpsURLConnection.setDefaultHostnameVerifier(hostNameVerifier);

      javax.net.ssl.TrustManager[] trustAllCerts = new javax.net.ssl.TrustManager[1];
      javax.net.ssl.TrustManager tm = new TrustAllTrustManager();
      trustAllCerts[0] = tm;
      javax.net.ssl.SSLContext sc = null;

      try {
         sc = javax.net.ssl.SSLContext.getInstance("SSL");
      } catch (NoSuchAlgorithmException e) {
         _logger.info(e);
      }

      javax.net.ssl.SSLSessionContext sslsc = sc.getServerSessionContext();
      sslsc.setSessionTimeout(0);
      try {
         sc.init(null, trustAllCerts, null);
      } catch (KeyManagementException e) {
         _logger.info(e);
      }
      javax.net.ssl.HttpsURLConnection.setDefaultSSLSocketFactory(
            sc.getSocketFactory());
   }

   /**
    *  Constructor used to inject the utility services (see the declaration
    *  in main/resources/spring/bundle-context-osgi.xml)
    *
    * @param userSessionService
    *    Service to access the current session information.
    * @param vimObjectReferenceService
    *    Service to access vSphere object references information.
    * @param registry
    *    Registration interface for data adapters and providers.
    */
   public VmDataProviderImpl(
            UserSessionService userSessionService,
            VimObjectReferenceService vimObjectReferenceService,
            DataServiceExtensionRegistry registry) {
      _userSessionService = userSessionService;
      _vimObjectReferenceService = vimObjectReferenceService;
      registry.registerDataAdapter(this, getProvidedTypeInfos());
   }

   /**
    * @return the types and properties handled by this adapter.
    */
   private TypeInfo[] getProvidedTypeInfos() {
      TypeInfo vmTypeInfo = new TypeInfo();
      vmTypeInfo.type = VM_TYPE;
      vmTypeInfo.properties = new String[] { VMDATA_PROPERTY };

      return new TypeInfo[] { vmTypeInfo };
   }

   /* (non-Javadoc)
    * @see com.vmware.vise.security.ClientSessionEndListener#sessionEnded(java.lang.String)
    */
   @Override
   public void sessionEnded(String clientId) {
      _logger.info("Logging out client session - " + clientId);
      // Clean up all session specific resources.
      // Logout from any session specific services.
   }

   /* (non-Javadoc)
    * @see com.vmware.vise.data.query.PropertyProviderAdapter#getProperties
    * (com.vmware.vise.data.query.PropertyRequestSpec)
    */
   @Override
   public ResultSet getProperties(PropertyRequestSpec propertyRequest) {
      // This is the main entry point of the PropertyProviderAdapter.
      // propertyRequest.properties contains the properties to be retrieved
      // which is just the "vmData" model name in our sample. We could also
      // have passed multiple properties and avoid using the VmData model.

      // Note: validation of propertyRequest is not really necessary as this simple
      // PropertyProviderAdapter is only called for the property it was registered for.

      ResultSet result = new ResultSet();

      try {
         List<ResultItem> resultItems = new ArrayList<ResultItem>();

         // A PropertyProviderAdapter can handle multiple objects, although
         // the vsphere-wssdk-ui client only uses it for one VM at a time.
         for (Object vmRef: propertyRequest.objects) {
            ResultItem resultItem = getVmProperties(vmRef);
            if (resultItem != null) {
               resultItems.add(resultItem);
            }
         }
         result.items = resultItems.toArray(new ResultItem[] {});

         // Note: result.totalMatchedObjectCount remains 0 in a PropertyProviderAdapter
         // since no new objects are discovered, we are just returning properties of
         // existing objects.

      } catch (Exception e) {
         _logger.error("VmDataProviderImpl.getProperties error: " + e);
         // Passing the exception in the result allows to display an error notification
         // in the client UI.
         result.error = e;
      }
      return result;
   }

   /**
    * Get the properties of an individual VM, in this case the ones
    * defined in the VmData model.
    *
    * @param vmRef
    *             The object reference passed by the client
    * @return ResultItem
    *             The result for that VM.
    *
    * @throws InvalidPropertyFaultMsg
    * @throws RuntimeFaultFaultMsg
    */
   private ResultItem getVmProperties(Object vmRef)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {

      // Create a vim25 ManagedObjectReference for the VirtualMachine
      ManagedObjectReference vmMor = new ManagedObjectReference();
      vmMor.setType(_vimObjectReferenceService.getResourceObjectType(vmRef));
      vmMor.setValue(_vimObjectReferenceService.getValue(vmRef));

      // Initialize the ServiceContent interface.
      // TODO: this should be optimized further in case this class is going to be
      // used for multiple VMs at a time, i.e. you can keep a map of serverGuid
      // to ServiceContent in order to avoid calling getServiceContent each time.
      String serverGuid = _vimObjectReferenceService.getServerGuid(vmRef);
      ServiceContent service = getServiceContent(serverGuid);
      if (service == null) {
         // We could also report an error here.
         return null;
      }

      VirtualMachineConfigInfo config = null;
      VmData vmData = new VmData();

      // Create Datacenter Property Spec
      com.vmware.vim25.PropertySpec propertySpec = new com.vmware.vim25.PropertySpec();
      propertySpec.setAll(Boolean.FALSE);
      propertySpec.setType("Datacenter");
      propertySpec.getPathSet().add("name");

      // Create VirtualMachine Property Spec
      com.vmware.vim25.PropertySpec propertySpec2 = new com.vmware.vim25.PropertySpec();
      propertySpec2.setAll(Boolean.FALSE);
      propertySpec2.setType(VM_TYPE);
      propertySpec2.getPathSet().add("config");

      // Create VirtualMachine Object Spec
      ObjectSpec objectSpec = new ObjectSpec();
      objectSpec.setObj(vmMor);
      objectSpec.setSkip(Boolean.FALSE);
      objectSpec.getSelectSet().addAll(buildTraversalSpecForVMToDatacenter());

      // Create PropertyFilterSpec using previous PropertySpec and ObjectSpec
      PropertyFilterSpec propertyFilterSpec = new PropertyFilterSpec();
      propertyFilterSpec.getPropSet().add(propertySpec);
      propertyFilterSpec.getPropSet().add(propertySpec2);
      propertyFilterSpec.getObjectSet().add(objectSpec);

      List<PropertyFilterSpec> propertyFilterSpecs = new ArrayList<PropertyFilterSpec>();
      propertyFilterSpecs.add(propertyFilterSpec);

      // Retrieve properties with the PropertyCollector
      List<ObjectContent> objectContents = _vimPort.retrieveProperties(
                  service.getPropertyCollector(), propertyFilterSpecs);
      if (objectContents != null) {
         for (ObjectContent content : objectContents) {
            if (VM_TYPE.equalsIgnoreCase(content.getObj().getType())) {
               List<DynamicProperty> dps = content.getPropSet();
               if (dps != null) {
                  for (DynamicProperty dp : dps) {
                     config = (VirtualMachineConfigInfo) dp.getVal();
                  }
               }
            } else {
               List<DynamicProperty> dps = content.getPropSet();
               if (dps != null) {
                  for (DynamicProperty dp : dps) {
                     vmData.datacenterName = (String) dp.getVal();
                     break;
                  }
               }
            }
         }
      }
      long size = 0;
      if (config != null) {
         List<VirtualDevice> vDevices = config.getHardware().getDevice();
         for (VirtualDevice vDevice : vDevices) {
            if (vDevice instanceof VirtualDisk) {
               size += ((VirtualDisk) vDevice).getCapacityInKB();
            }
         }
      }

      // Note: these strings should be localized in a real application
      vmData.capacityInKb = size + " KB";
      vmData.numberOfVirtualCpus = config.getHardware().getNumCPU() + " CPU";

      ResultItem ri = new ResultItem();
      ri.resourceObject = vmRef;
      PropertyValue pv = new PropertyValue();
      pv.resourceObject = vmRef;
      pv.propertyName = VMDATA_PROPERTY;
      pv.value = vmData;
      ri.properties = new PropertyValue[] { pv };

      return ri;
   }

   /**
    *
    * @return An array of SelectionSpec to navigate from the VM and move
    *         upwards to reach the Datacenter
    */
   private static List<SelectionSpec> buildTraversalSpecForVMToDatacenter() {

      // For Folder -> Folder recursion
      SelectionSpec sspecvfolders = new SelectionSpec();
      sspecvfolders.setName("VisitFolders");

      TraversalSpec visitFolders = new TraversalSpec();
      visitFolders.setType("Folder");
      visitFolders.setPath("parent");
      visitFolders.setSkip(Boolean.FALSE);
      visitFolders.setName("VisitFolders");
      visitFolders.getSelectSet().add(sspecvfolders);

      // For vApp -> vApp recursion
      SelectionSpec sspecvApp = new SelectionSpec();
      sspecvApp.setName("vAppToVApp");

      SelectionSpec sspecvAppToFolder = new SelectionSpec();
      sspecvAppToFolder.setName("vAppToFolder");

      // For vApp -> parent folder
      TraversalSpec vAppToFolder = new TraversalSpec();
      vAppToFolder.setType("VirtualApp");
      vAppToFolder.setPath("parentFolder");
      vAppToFolder.setSkip(Boolean.FALSE);
      vAppToFolder.setName("vAppToFolder");
      vAppToFolder.getSelectSet().add(sspecvfolders);

      // For vApp -> parent vApp
      TraversalSpec vAppToVApp = new TraversalSpec();
      vAppToVApp.setType("VirtualApp");
      vAppToVApp.setPath("parentVApp");
      vAppToVApp.setSkip(Boolean.FALSE);
      vAppToVApp.setName("vAppToVApp");
      vAppToVApp.getSelectSet().add(sspecvApp);
      vAppToVApp.getSelectSet().add(sspecvAppToFolder);

      // For VM -> parent vApp
      TraversalSpec vmTovApp = new TraversalSpec();
      vmTovApp.setType(VM_TYPE);
      vmTovApp.setPath("parentVApp");
      vmTovApp.setSkip(Boolean.FALSE);
      vmTovApp.setName("vmTovApp");
      vmTovApp.getSelectSet().add(vAppToVApp);
      vmTovApp.getSelectSet().add(vAppToFolder);

      // For VM -> parent folder
      TraversalSpec vmToFolder = new TraversalSpec();
      vmToFolder.setType(VM_TYPE);
      vmToFolder.setPath("parent");
      vmToFolder.setSkip(Boolean.FALSE);
      vmToFolder.setName("vmToFolder");
      vmToFolder.getSelectSet().add(sspecvfolders);

      List<SelectionSpec> speclist = new ArrayList<SelectionSpec>();
      speclist.add(vmToFolder);
      speclist.add(vmTovApp);
      speclist.add(visitFolders);
      return speclist;
   }

   /**
    * Find the relevant ServerInfo Object for the given serverGuid and the
    * current session.
    */
   private ServerInfo getServerInfoObject(String serverGuid) {
      UserSession userSession = _userSessionService.getUserSession();

      for (ServerInfo sinfo : userSession.serversInfo) {
         if (sinfo.serviceGuid.equalsIgnoreCase(serverGuid)) {
            return sinfo;
         }
      }
      return null;
   }


   /**
    * Get the ServiceContent for a given VC server.
    */
   private ServiceContent getServiceContent(String serverGuid) {

      ServerInfo serverInfoObject = getServerInfoObject(serverGuid);
      String sessionCookie = serverInfoObject.sessionCookie;
      String serviceUrl = serverInfoObject.serviceUrl;

      if (_logger.isDebugEnabled()) {
         _logger.debug("getServiceContent: sessionCookie = "+ sessionCookie +
                        ", serviceUrl= " + serviceUrl);
      }

      List<String> values = new ArrayList<String>();
      values.add("vmware_soap_session=" + sessionCookie);
      Map<String, List<String>> reqHeadrs = new HashMap<String, List<String>>();
      reqHeadrs.put("Cookie", values);

      Map<String, Object> reqContext = ((BindingProvider)_vimPort).getRequestContext();
      reqContext.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, serviceUrl);
      reqContext.put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);
      reqContext.put(MessageContext.HTTP_REQUEST_HEADERS, reqHeadrs);

      final ManagedObjectReference svcInstanceRef = new ManagedObjectReference();
      svcInstanceRef.setType(SERVICE_INSTANCE);
      svcInstanceRef.setValue(SERVICE_INSTANCE);

      ServiceContent serviceContent = null;
      try {
         serviceContent = _vimPort.retrieveServiceContent(svcInstanceRef);
      } catch (RuntimeFaultFaultMsg e) {
         _logger.error("getServiceContent error: " + e);
      }

      return serviceContent;
   }

   /**
    * The following code snippet will trust all certificates Used only in
    * testing.
    */
   private static class TrustAllTrustManager implements
      javax.net.ssl.TrustManager, javax.net.ssl.X509TrustManager {

      @Override
      public java.security.cert.X509Certificate[] getAcceptedIssuers() {
         return null;
      }

      @Override
      public void checkServerTrusted(java.security.cert.X509Certificate[] certs,
            String authType) throws java.security.cert.CertificateException {
         return;
      }

      @Override
      public void checkClientTrusted(java.security.cert.X509Certificate[] certs,
            String authType) throws java.security.cert.CertificateException {
         return;
      }
   } // End class TrustAllTrustManager.

} // End class VmPropertyProviderImpl.