/* Copyright 2012 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.chassisa;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.vmware.samples.chassisa.model.Chassis;
import com.vmware.vise.data.Constraint;
import com.vmware.vise.data.PropertySpec;
import com.vmware.vise.data.ResourceSpec;
import com.vmware.vise.data.query.Comparator;
import com.vmware.vise.data.query.CompositeConstraint;
import com.vmware.vise.data.query.DataProviderAdapter;
import com.vmware.vise.data.query.ObjectIdentityConstraint;
import com.vmware.vise.data.query.PropertyConstraint;
import com.vmware.vise.data.query.PropertyValue;
import com.vmware.vise.data.query.QuerySpec;
import com.vmware.vise.data.query.RequestSpec;
import com.vmware.vise.data.query.Response;
import com.vmware.vise.data.query.ResultItem;
import com.vmware.vise.data.query.ResultSet;
import com.vmware.vise.data.query.type;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Custom data adapter used to add Chassis objects into vSphere client.
 */
// declares the supported chassis type, avoids conflicts with other chassis types
@type("samples:ChassisA")
public class ChassisDataAdapter implements DataProviderAdapter {
   /* Object types must be qualified with a namespace. */
   public static final String CHASSIS_TYPE = "samples:ChassisA";

   private final VimObjectReferenceService _objectRefService;
   private final ObjectStore _objectStore;

   /**
    * Constructor.
    *
    * @param objectReferenceService
    *    Injection of the VimObjectReferenceService to handle non-vSphere objects
    *    references because this simple implementation doesn't worry about namespace.
    *
    *    NOTE: A real implementation should use its own type resolver as shown in the
    *    more advanced version chassisRackVSphere-service.
    *
    * @param objectStore The in-memory store used for this sample.
    *
    * @see com.vmware.vise.vim.data.VimObjectReferenceService
    */
   public ChassisDataAdapter(
         VimObjectReferenceService objectReferenceService, ObjectStore objectStore) {
      _objectRefService = objectReferenceService;
      _objectStore = objectStore;

      // The constructor should not do any significant data initialization:
      // - There is a risk of slowing down the bundle deployment.
      // - It is not practical to report data setup errors here.
      // - In most cases the back-end server parameters are not known at this time.
   }

   //------------------------------------------------------------------------------------
   // DataProviderAdapter methods.

   @Override
   /**
    * Hook into vSphere client's DataService: all query requests for the types
    * supported by this adapter land here.  Depending on the request the response
    * will contain newly discovered objects, or properties on existing objects.
    *
    * @see com.vmware.vise.data.query.DataProviderAdapter#getData(
    *    com.vmware.vise.data.query.RequestSpec)
    */
   public Response getData(RequestSpec request) {
      if (request == null) {
         throw new IllegalArgumentException("request must be non-null.");
      }

      // The platform may be combining several queries in one call.
      // A real adapter should transform these queries into calls to a remote
      // plugin server to fetch data. In this sample we process everything in memory.
      QuerySpec[] querySpecs = request.querySpec;
      List<ResultSet> results = new ArrayList<ResultSet>(querySpecs.length);
      for (QuerySpec qs : querySpecs) {
         ResultSet rs = processQuery(qs);
         results.add(rs);
      }

      Response response = new Response();
      response.resultSet = results.toArray(new ResultSet[]{});
      return response;
   }

   //------------------------------------------------------------------------------------
   // Private methods.


   /**
    * Process a single QuerySpec.
    */
   private ResultSet processQuery(QuerySpec qs) {
      ResultSet rs = new ResultSet();

      // Validate the query
      if (!validateQuerySpec(qs)) {
         // Ignore queries that cannot be handled by this adapter. Don't throw
         // an error in this case to avoid affecting the UI.
         return rs;
      }

      // Process the constraint.
      List<ResultItem> items = processConstraint(
            qs.resourceSpec.constraint,
            qs.resourceSpec.propertySpecs);

      // Return a ResultSet containing the items:
      // - Note that totalMatchedObjectCount is only relevant when the query is
      //   for discovering new objects (i.e. constraint is of type Constraint)
      // - This sample doesn't handle paging and sorting of items but your plugin should!
      //   See chassisRackVSphere-service for paging & sorting code.
      rs.totalMatchedObjectCount = (items != null) ? items.size() : 0;
      rs.items = items.toArray(new ResultItem[]{});
      rs.queryName = qs.name;
      return rs;
   }

   private String[] convertPropertySpec(PropertySpec[] pSpecs) {
      Set<String> properties = new HashSet<String>();
      if (pSpecs != null) {
         for (PropertySpec pSpec : pSpecs) {
            for (String property : pSpec.propertyNames) {
               properties.add(property);
            }
         }
      }
      return properties.toArray(new String[]{});
   }

   /**
    * Process a particular constraint and return the ResultItem list for that
    * constraint (this can be nested)
    *
    * Note: this version supports a limited set of constraints to keep things simple,
    * see chassisRackVSphere-service for a more complete implementation.
    */
   private List<ResultItem> processConstraint(
         Constraint constraint,
         PropertySpec[] propertySpecs) {
      List<ResultItem> items = null;

      if (constraint instanceof ObjectIdentityConstraint) {
         // Constraint for getting properties of a specific chassis
         // i.e. the matching criteria is on the object itself.
         ObjectIdentityConstraint oic = (ObjectIdentityConstraint)constraint;
         items = processObjectIdentityConstraint(oic, propertySpecs);
      } else if (constraint instanceof CompositeConstraint) {
         // Constraint is a composite of nested constraints
         CompositeConstraint cc = (CompositeConstraint)constraint;
         items = processCompositeConstraint(cc, propertySpecs);
      } else if (constraint instanceof PropertyConstraint) {
         // Constraint where the matching criteria is based on properties of
         // the target objects.
         PropertyConstraint pc = (PropertyConstraint)constraint;
         items = processPropertyConstraint(pc, propertySpecs);
      } else if (isSimpleConstraint(constraint)) {
         // Simple constraint with only a targetType, which means that the
         // adapter should return all objects of that type.
         items = processSimpleConstraint(constraint, propertySpecs);
      }
      return items;
   }

   /**
    * Handles a CompositeConstraint assuming that the conjoiner is 'OR' and delegating
    * the processing of each constraint to the appropriate handler.
    */
   private List<ResultItem> processCompositeConstraint(
         CompositeConstraint cc,
         PropertySpec[] propertySpecs) {
      List<ResultItem> items = new ArrayList<ResultItem>();
      for (Constraint constraint : cc.nestedConstraints) {
         List<ResultItem> individualItems =
               processConstraint(constraint, propertySpecs);
         items.addAll(individualItems);
      }
      return items;
   }

   /**
    * Returns all objects matching the targetType specified in the constraint
    * (returns all chassis objects in this case)
    */
   private List<ResultItem> processSimpleConstraint(
         Constraint constraint,
         PropertySpec[] propertySpecs) {

      List<ResultItem> items = new ArrayList<ResultItem>();
      String[] requestedProperties = convertPropertySpec(propertySpecs);

      // Get a snapshot of the current chassis list.
      // An implementation with a large data set should handle paging.
      Map<String, Chassis> currentObjects = _objectStore.getObjects();
      Iterator<String> i = currentObjects.keySet().iterator();
      while(i.hasNext()) {
         String uid = i.next();
         Chassis chassis = currentObjects.get(uid);

         // Create chassis result item
         ResultItem ri = createChassisResultItem(uid, chassis, requestedProperties);
         if (ri != null) {
            items.add(ri);
         }
      }

      return items;
   }

   /**
    * Returns a ResultItem containing the matching Chassis object
    * and its requested properties.
    */
   private List<ResultItem> processObjectIdentityConstraint(
         ObjectIdentityConstraint constraint,
         PropertySpec[] propertySpecs) {

      List<ResultItem> items = new ArrayList<ResultItem>();
      String[] requestedProperties = convertPropertySpec(propertySpecs);

      // constraint.target is the chassis reference passed by the UI, so we can access
      // its uid and the actual chassis object, if it still exists in the back-end.
      String uid = _objectRefService.getUid(constraint.target);
      Chassis chassis = _objectStore.getChassis(uid);

      if (chassis != null) {
         ResultItem ri = createChassisResultItem(uid, chassis, requestedProperties);
         if (ri != null) {
            items.add(ri);
         }
      }
      return items;
   }

   /**
    * Handle a PropertyConstraint with a textual comparator
    *
    * @return a ResultItem list with the objects for which the value of
    *    pc.propertyName is a String and is equal to pc.comparableValue
    */
   private List<ResultItem> processPropertyConstraint(
         PropertyConstraint pc,
         PropertySpec[] propertySpecs) {
      assert (pc.comparator == Comparator.EQUALS);
      String comparableValue = pc.comparableValue.toString();
      List<ResultItem> items = new ArrayList<ResultItem>();
      String[] requestedProperties = convertPropertySpec(propertySpecs);

      // Get a snapshot of the current chassis list to compare data.
      // A real implementation would do that on the back-end server.
      Map<String, Chassis> currentObjects = _objectStore.getObjects();
      Iterator<String> i = currentObjects.keySet().iterator();

      while(i.hasNext()) {
         String uid = i.next();
         Chassis chassis = currentObjects.get(uid);

         if (comparableValue.equals(chassis.getProperty(pc.propertyName))) {
            ResultItem ri = createChassisResultItem(uid, chassis, requestedProperties);
            if (ri != null) {
               items.add(ri);
            }
         }
      }
      return items;
   }

   /**
    * Prepare the result for a particular chassis and the properties to be fetched.
    *
    * @param uid  The chassis uid
    * @param chassis The chassis object
    * @param requestedProperties The list of property names
    * @return the ResultItem for the given chassis and properties
    */
   private ResultItem createChassisResultItem(
         String uid, Chassis chassis, String[] requestedProperties) {

      ResultItem ri = new ResultItem();

      // Important: the ResultItem resourceObject is the chassis resource reference
      // not the Chassis object itself!  In practice it will be a URI that gets mapped
      // into a IResourceReference in the UI layer.
      Object chassisRef = chassis.getChassisReference();
      ri.resourceObject = chassisRef;

      List<PropertyValue> propValArr = new ArrayList<PropertyValue>(requestedProperties.length);
      for(int i = 0; i < requestedProperties.length; ++i) {
         String requestedProperty = requestedProperties[i];
         Object value = chassis.getProperty(requestedProperty);

         // Queries may include properties unsupported by this adapter,
         // it is recommended to skip them instead of returning a null value.
         if (value != null) {
            PropertyValue pv = new PropertyValue();
            pv.resourceObject = chassisRef;
            pv.propertyName = requestedProperty;
            pv.value = value;
            propValArr.add(pv);
         }
      }
      ri.properties = propValArr.toArray(new PropertyValue[0]);
      return ri;
   }

   /**
    * Validates the input query spec.
    *
    * @return
    *    Returns false if the query spec cannot be processed by this data adapter.
    */
   private boolean validateQuerySpec(QuerySpec qs) {
      if (qs == null) {
         return false;
      }

      ResourceSpec resourceSpec = qs.resourceSpec;
      if (resourceSpec == null || resourceSpec.constraint == null) {
         return false;
      }
      return validateConstraint(resourceSpec.constraint);
   }

   /**
    * @param constraint the constraint from the current querySpec
    * @return true if we can process the constraint, false otherwise
    *
    * Note: this version supports only a few constraints to keep things simple,
    * see also chassisRackVSphere-service.
    */
   private boolean validateConstraint(Constraint constraint) {
      if (constraint instanceof ObjectIdentityConstraint) {
         // We only deal with constraints on chassis objects
         Object source = ((ObjectIdentityConstraint)constraint).target;
         return (source != null &&
               CHASSIS_TYPE.equals(_objectRefService.getResourceObjectType(source)));

      } else if (constraint instanceof CompositeConstraint) {
         CompositeConstraint cc = (CompositeConstraint)constraint;
         for (Constraint c : cc.nestedConstraints) {
            if (!validateConstraint(c)) {
               return false;
            }
         }
         return true;

      } else if (constraint instanceof PropertyConstraint) {
         // Only handles constraint for text comparisons
         return CHASSIS_TYPE.equals(constraint.targetType) &&
               ((PropertyConstraint)constraint).comparator == Comparator.EQUALS;

      } else if (isSimpleConstraint(constraint)) {
         // we only return chassis objects
         return (CHASSIS_TYPE.equals(constraint.targetType));
      }
      // For any other constraint - mark the query as invalid.
      return false;
   }


   /**
    * @return true if constraint has the type Constraint, not a sub-type.
    */
   private boolean isSimpleConstraint(Object constraint) {
      // Note that a comparison on the class itself would fail when the class
      // loader is different.
      return (constraint.getClass().getSimpleName().equals(
            Constraint.class.getSimpleName()));
   }

}
