/** Copyright 2013 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.chassisa;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.vmware.samples.chassisa.model.Chassis;
import com.vmware.samples.chassisa.model.ChassisInfo;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Implementation of ChassisService.
 *
 */
public class ChassisServiceImpl implements ChassisService {
   private static final Log _logger = LogFactory.getLog(ChassisServiceImpl.class);

   private final VimObjectReferenceService _objectRefService;
   private final ObjectStore _objectStore;

   /**
    * Constructor.
    *
    * @param objectReferenceService
    *    Injection of the VimObjectReferenceService to handle non-vSphere objects
    *    references because this simple implementation doesn't worry about namespace.
    *
    *    NOTE: A real implementation should use its own type resolver as shown in the
    *    more advanced version chassisRackVSphere-service.
    *
    * @param objectStore The in-memory store used for this sample.
    *
    * @see com.vmware.vise.vim.data.VimObjectReferenceService
    */
   public ChassisServiceImpl(
         VimObjectReferenceService objectReferenceService, ObjectStore objectStore) {
      _objectRefService = objectReferenceService;
      _objectStore = objectStore;
   }

   //------------------------------------------------------------------------------------
   // ChassisService methods.

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisservice.ChassisService#createChassis(com.vmware.samples.chassisservice.model.ChassisInfo)
    */
   @Override
   public Object createChassis(ChassisInfo chassisInfo) {
      Chassis chassis = _objectStore.createChassis(chassisInfo, true);

      if (chassis == null) {
         _logger.info("Chassis not created because this name is taken: " + chassisInfo.name);
         return null;
      }
      _logger.info("Chassis created: " + chassis.toString());

      // Important: the returned object is the reference generated by the reference service
      // not the chassis object itself!  In practice it will be a URI that get mapped into
      // a IResourceReference in the UI layer.
      return chassis.getChassisReference();
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisservice.ChassisService#udpateChassis(java.lang.Object)
    */
   @Override
   public boolean updateChassis(Object chassisRef, ChassisInfo newInfo) {
      String uid = _objectRefService.getUid(chassisRef);

      Chassis chassis = _objectStore.getChassis(uid);
      if (chassis == null) {
         // Chassis may have already been deleted by another user
         _logger.info("Chassis not found during edit operation: " + uid);
         return false;
      }

      return _objectStore.updateChassis(uid, newInfo, chassisRef);
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisservice.ChassisService#deleteChassis(java.lang.Object)
    */
   @Override
   public boolean deleteChassis(Object chassisRef) {
      String uid = _objectRefService.getUid(chassisRef);

      Chassis chassis = _objectStore.removeChassis(uid);
      if (chassis == null) {
         // Chassis may have already been deleted by another user
         _logger.info("Chassis not found during delete operation: " + uid);
         return false;
      }
      return true;
   }

}
