/** Copyright 2013 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.chassisa.model;

/**
 * Chassis infrastructure for housing multiple servers.
 *
 * NOTES:
 * - Chassis objects are immutable and de-facto thread safe.
 * - Real implementations should use immutable Java DTOs populated by the back-end.
 * - It is important to understand the difference between the Chassis object and
 *    its reference used to represent the object in the UI.
 */
public class Chassis {

   /* Object property names, must match the names used in the client's requests */
   private static final String CHASSIS_NAME = "name";
   private static final String CHASSIS_SERVER_TYPE = "serverType";
   private static final String CHASSIS_DIMENSIONS = "dimensions";

   private final String _name;
   private final String _dimensions;
   private final String _serverType;

   // Object reference representing a chassis in the UI. In practice this will be a URI.
   // For a more complete implementation see the chassisRackVSphere-service sample.
   private final Object _chassisReference;

   /**
    * Constructor
    *
    * @param chassisInfo   The data used for the new chassis.
    * @param chassisRef    The object reference.
    */
   public Chassis(ChassisInfo chassisInfo, Object chassisRef) {
	   _name = chassisInfo.name;
	   _dimensions = chassisInfo.dimensions;
	   _serverType = chassisInfo.serverType;
	   _chassisReference = chassisRef;
   }

   /**
    * Name of the object.
    */
   public String getName() {
      return _name;
   }

   /**
    * Free form string representing the chassis dimensions
    */
   public String getDimensions() {
      return _dimensions;
   }

   /**
    * Free form string representing the type of server hosted by this chassis.
    */
   public String getServerType() {
      return _serverType;
   }

   /**
    * @return the chassis reference to be used in the UI.
    */
   public Object getChassisReference() {
	   return _chassisReference;
   }

   /**
    * Get a property value given its name.
    *
    * @param propName
    * @return the value for the given chassis property, or null
    *       if the property doesn't exist.
    */
   public String getProperty(String propName) {
      if (CHASSIS_NAME.equals(propName)) {
         return getName();
      }
      else if (CHASSIS_SERVER_TYPE.equals(propName)) {
         return getServerType();
      }
      else if (CHASSIS_DIMENSIONS.equals(propName)) {
         return getDimensions();
      }
      return null;
   }

   public String toString() {
      return "name: " + _name + ", serverType: " + _serverType +
            ", dimensions" + _dimensions;
   }
}
