/** Copyright 2013 VMware, Inc. All rights reserved. -- VMware Confidential */
package com.vmware.samples.chassisa.mvc;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.gson.Gson;
import com.vmware.samples.chassisa.ChassisDataAdapter;
import com.vmware.samples.chassisa.ChassisService;
import com.vmware.samples.chassisa.model.ChassisInfo;
import com.vmware.vise.data.query.ObjectReferenceService;

/**
 * A controller to serve HTTP JSON GET/POST requests to the endpoint "/actions.html".
 */
@Controller
@RequestMapping(value = "/actions.html")
public class ActionsController {
   private final static Log _logger = LogFactory.getLog(ActionsController.class);

   // UI plugin resource bundle for localized messages
   private final String RESOURCE_BUNDLE = "com_vmware_samples_chassisa";

   private final ChassisService _chassisService;
   private final ObjectReferenceService _objectReferenceService;

   @Autowired
   public ActionsController(
         ChassisService chassisService,
         @Qualifier("objectReferenceService") ObjectReferenceService objectReferenceService) {
      _chassisService = chassisService;
      _objectReferenceService = objectReferenceService;
      QueryUtil.setObjectReferenceService(objectReferenceService);
   }


   /**
    * Generic method to invoke an action on a given object or a global action.
    *
    * @param actionUid  the action Uid as defined in plugin.xml
    *
    * @param targets  null for a global action, comma-separated list of object ids
    *    for an action on 1 or more objects
    *
    * @param json additional data in JSON format, or null for the delete action.
    *
    * @return
    *    Returns a map that will be converted to a Json string.
    */
   @RequestMapping(method = RequestMethod.POST)
   @ResponseBody
   public Map<String, Object> invoke(
            @RequestParam(value = "actionUid", required = true) String actionUid,
            @RequestParam(value = "targets", required = false) String targets,
            @RequestParam(value = "json", required = false) String json)
            throws Exception {

      // Parameters validation
      Object objectRef = null;
      if (targets != null) {
         String[] objectIds = targets.split(",");
         if (objectIds.length > 1) {
            // Chassis actions only support 1 target object
            _logger.warn("Action " + actionUid + " called with " + objectIds.length
                  + " target objects, will use only the first one");
         }
         String objectId = ObjectIdUtil.decodeParameter(objectIds[0]);
         objectRef = _objectReferenceService.getReference(objectId);
         if (objectRef == null) {
            _logger.error("Object not found with id: " + objectId);
            // TODO handle error
         }
      }
      ChassisInfo chassisInfo = null;
      if (json != null) {
         // Create a ChassisInfo java object from the json data.
         Gson gson = new Gson();
         chassisInfo = gson.fromJson(json, ChassisInfo.class);
      } else if (!actionUid.equals("com.vmware.samples.chassisa.deleteChassis")) {
         _logger.error("Missing json data for " + actionUid);
         // TODO handle error
      }

      ActionResult actionResult = new ActionResult(actionUid, RESOURCE_BUNDLE);

      if (actionUid.equals("com.vmware.samples.chassisa.editChassis")) {
         boolean result = _chassisService.updateChassis(objectRef, chassisInfo);
         actionResult.setObjectChangedResult(result, "editAction.notFound");

      } else if (actionUid.equals("com.vmware.samples.chassisa.deleteChassis")) {
         boolean result = _chassisService.deleteChassis(objectRef);
         actionResult.setObjectDeletedResult(result, "deleteAction.notFound");

      } else if (actionUid.equals("com.vmware.samples.chassisa.createChassis")) {
         Object result = _chassisService.createChassis(chassisInfo);
         if (result != null) {
            actionResult.setObjectAddedResult((URI)result, ChassisDataAdapter.CHASSIS_TYPE, null);
         } else {
            // Case where the name is already taken
            String[] params = new String[] { chassisInfo.name };
            actionResult.setErrorMessage("createAction.nameTaken", params);
         }

      } else {
         String warning = "Action not implemented yet! "+ actionUid;
         _logger.warn(warning);
         actionResult.setErrorLocalizedMessage(warning);
      }
      return actionResult.getJsonMap();
   }

   /**
    * Generic handling of internal exceptions.
    * Sends a 500 server error response along with a json body with messages
    *
    * @param ex The exception that was thrown.
    * @param response
    * @return a map containing the exception message, the cause, and a stackTrace
    */
   @ExceptionHandler(Exception.class)
   @ResponseBody
   public Map<String, String> handleException(Exception ex, HttpServletResponse response) {
      response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());

      Map<String,String> errorMap = new HashMap<String,String>();
      errorMap.put("message", ex.getMessage());
      if(ex.getCause() != null) {
         errorMap.put("cause", ex.getCause().getMessage());
      }
      StringWriter sw = new StringWriter();
      PrintWriter pw = new PrintWriter(sw);
      ex.printStackTrace(pw);
      errorMap.put("stackTrace", sw.toString());

      return errorMap;
   }
}

