/** Copyright 2012 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.chassisb.model;

import com.vmware.samples.chassisb.model.ChassisInfo;
import com.vmware.samples.chassisb.model.Rack;
import com.vmware.vise.data.query.ObjectReferenceService;

/**
 * Chassis infrastructure for housing multiple servers.
 * A chassis can be associated to one rack and it can house one or more hosts.
 *
 * NOTES:
 * - Chassis objects are not completely immutable because of their hosts list but
 *    this implementation should be thread safe.
 * - In a real implementation this would be an immutable Java DTO populated by
 *    the back-end server or database layer.
 * - It is important to understand the difference between the Chassis object and
 *    its URI reference used to represent the object in the UI.
 */
public class Chassis extends ModelObject {

   private final String _name;
   private final String _dimensions;
   private final String _serverType;

   /**
    * Rack enclosure for this chassis, or null.
    */
   private final Rack _rack;

   /**
    * vSphere hosts running on this chassis.
    * @GuardedBy _hostsLock
    */
   private Object[] _hosts = new Object[0];

   private Object _hostsLock = new Object();

   /**
    * Constructor.
    *
    * @param chassisInfo   Chassis data.
    * @param rack          Related rack, or null.
    * @param id            Resource id.
    */
   public Chassis(ChassisInfo chassisInfo, Rack rack, String id) {
      this.setId(id);
      this._name = chassisInfo.name;
      this._dimensions = chassisInfo.dimensions;
      this._serverType = chassisInfo.serverType;
      this._rack = rack;
   }

   /**
    * Name of the object.
    */
   public String getName() {
      return _name;
   }

   /**
    * Free form string representing the chassis dimensions
    */
   public String getDimensions() {
      return _dimensions;
   }

   /**
    * Free form string representing the type of server hosted by this chassis.
    */
   public String getServerType() {
      return _serverType;
   }

   /**
    * The rack enclosure for this chassis, or null if no relation exists.
    */
   public Rack getRack() {
      return _rack;
   }

   /**
    * The vSphere hosts running on this chassis.
    * (We don't know the host object type, those are internal vSphere objects)
    */
   public Object[] getHosts() {
      // Since the _hosts list may be modified by another thread we can only
      // return a snapshot of the current list.
      synchronized(_hostsLock) {
         Object hostsCopy[] = new Object[_hosts.length];
         System.arraycopy(_hosts, 0, hostsCopy, 0, _hosts.length);
         return hostsCopy;
      }
   }

   @Override
   public Object getProperty(String property) {
      if ("name".equals(property)) {
         return getName();
      } else if ("serverType".equals(property)) {
         return getServerType();
      } else if ("dimensions".equals(property)) {
         return getDimensions();
      } else if ("rack".equals(property)) {
         return getRack();
      } else if ("host".equals(property)) {
         return getHosts();
      }
      return UNSUPPORTED_PROPERTY;
   }

   /**
    * Attempt to remove a host from the list of hosts related to this chassis.
    *
    * @param hostRef the target host reference
    * @param objRefService the service allowing to extract an object reference uid
    * @return true if the host was found, false otherwise.
    */
   public boolean removeHost(Object hostRef, ObjectReferenceService objRefService) {
      synchronized(_hostsLock) {
         if (_hosts == null || _hosts.length == 0) {
            return false;
         }
         String hostUid = objRefService.getUid(hostRef);
         if (hostUid == null) {
            return false;
         }

         for (Object host: _hosts) {
            // Look for the host with the same uid and remove it for this chassis
            if (hostUid.equals(objRefService.getUid(host))) {
               removeHost(host);
               return true;
            }
         }
         return false;
      }
   }

   /**
    * Add a host to a chassis.
    *
    * @param newHost the host object to add
    * @param _objRefService the service allowing to extract an object reference uid
    * @return true if the host gets added, false if it was already present.
    */
   public boolean addHost(Object newHost, ObjectReferenceService objRefService) {
      synchronized(_hostsLock) {
         if (_hosts == null || _hosts.length == 0) {
            _hosts = new Object[] { newHost };
            return true;
         }

         String hostUid = objRefService.getUid(newHost);
         if (hostUid == null) {
            return false;
         }
         for (Object host: _hosts) {
            if (hostUid.equals(objRefService.getUid(host))) {
               // newHost is already added to this chassis
               return false;
            }
         }
         addHost(newHost);
         return true;
      }
   }

   /**
    * @GuardedBy _hostsLock
    */
   private void addHost(Object newHost) {
      // Add one element to the _hosts array
      Object[] hosts = new Object[_hosts.length + 1];
      for (int i = 0; i < _hosts.length; i++) {
         hosts[i] = _hosts[i];
      }
      hosts[_hosts.length] = newHost;
      _hosts = hosts;
   }

   /**
    * @GuardedBy _hostsLock
    */
   private void removeHost(Object oldHost) {
      // Clone the _hosts array with one less element
      Object[] hosts = new Object[_hosts.length - 1];
      boolean found = false;
      for (int i = 0; i < hosts.length; i++) {
         if (!found && (_hosts[i] == oldHost)) {
            found = true;
         } else {
            hosts[i] = _hosts[i + (found ? 1 : 0)];
         }
      }
      _hosts = hosts;
   }

   // Override equals and hashCode to be able to compare chassis in lists

   /* (non-Javadoc)
    * @see java.lang.Object#equals(java.lang.Object)
    */
   @Override
   public boolean equals(Object o) {
      if (o == null || !(o instanceof Chassis)) {
         return false;
      }
      return getId().equals(((Chassis)o).getId());
   }

   /* (non-Javadoc)
    * @see java.lang.Object#hashCode()
    */
   @Override
   public int hashCode() {
      return getId().hashCode();
   }

}
