/** Copyright 2012 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.chassisb.model;

import java.util.ArrayList;
import java.util.List;

import com.vmware.samples.chassisb.model.Chassis;

/**
 * Rack enclosure for Chassis objects.
 *
 * NOTES:
 * - Rack objects are not completely immutable because of their chassis list but
 *    this implementation should be thread safe.
 * - In a real implementation Rack would be an immutable Java DTO populated by
 *    the back-end server or database layer.
 * - It is important to understand the difference between the Rack object and
 *    its URI reference used to represent the object in the UI.
 */
public class Rack extends ModelObject {

   private volatile String _name;
   private volatile String _location;

   // List of chassis related to this rack
   private final List<Chassis> _chassisList = new ArrayList<Chassis>();

   /**
    * Constructor.
    *
    * @param name       Rack name.
    * @param location   Rack location.
    * @param id         Resource id.
    */
   public Rack(String name, String location, String id) {
      this.setId(id);
      this._name = name;
      this._location = location;
   }

   /**
    * Name of the rack.
    */
   public String getName() {
      return _name;
   }

   /**
    * Free form string representing the location of the rack
    * (GPS coordinates in this sample)
    */
   public String getLocation() {
       return _location;
   }

   /**
    * Get the chassis objects located on this rack.
    *
    * @return the chassis list, or null if the rack contains no chassis
    */
   public List<Chassis> getChassis() {
      synchronized(_chassisList) {
         if (_chassisList.size() == 0) {
            // important to return null instead of an empty list because
            // null is interpreted as false if the query looks for a boolean.
            return null;
         }
         // use a copy of the chassis list, so the caller can iterate
         // through the list in a thread-safe manner
         return new ArrayList<Chassis>(_chassisList);
      }
   }


   @Override
   public Object getProperty(String property) {
      if ("name".equals(property)) {
         return getName();
      } else if ("location".equals(property)) {
         return getLocation();
      } else if ("chassis".equals(property)) {
         return getChassis();
      }
      return UNSUPPORTED_PROPERTY;
   }

   /**
    * Add a new chassis to the rack.
    *
    * @param newChassis
    * @return true if success, false if the chassis was already related to this rack.
    */
   public boolean addChassis(Chassis newChassis) {
      synchronized(_chassisList) {
         return _chassisList.add(newChassis);
      }
   }

   /**
    * Remove the relation between chassis to the rack.
    *
    * @param chassis the chassis to remove
    * @return true if success, false if the chassis was already related to this rack.
    */
   public boolean removeChassis(Chassis chassis) {
      synchronized(_chassisList) {
         return _chassisList.remove(chassis);
      }
   }

   /**
    * Replaces existing chassis in the rack with a new one.
    *
    * @param oldChassis the old chassis to replace
    * @param newChassis the new chassis to add as replacement
    */
   public void replaceChassis(Chassis oldChassis, Chassis newChassis) {
      synchronized(_chassisList) {
         int index = _chassisList.indexOf(oldChassis);
         if (index >= 0) {
            _chassisList.set(index, newChassis);
         }
      }
   }
}