/* Copyright 2011 VMware, Inc. All rights reserved. -- VMware Confidential */
package com.vmware.samples.globalview;

import com.vmware.samples.globalview.model.GlobalSettings;
import com.vmware.samples.globalview.model.SettingEnum;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Properties;

/**
 * Implementation of the GlobalService interface to read and write settings data.
 *
 * This implementation handles data persistence in a locale file settings.properties:
 * - The initial settings are read from a file included with the bundle resources.
 * - New settings are saved in a common location and are persistent across server restarts.
 *
 * Note: this is ok for light configuration data, but large or complex data sets should be
 * handled with a separate back-end server/database.
 */
public class GlobalServiceImpl implements GlobalService {
   private static final Log _logger = LogFactory.getLog(GlobalServiceImpl.class);

   /**
    * OS-dependent directory where to save persistent data.
    */
   private static final String SETTINGS_DIR_PATH = getGlobalViewDataFolder();

   /**
    * Properties file holding the setting values.
    * An XML format would be more appropriate for more complex settings.
    */
   private static final String SETTINGS_PROPERTIES = "settings.properties";
   private static final String SETTINGS_FILE_PATH =
         SETTINGS_DIR_PATH + "/" + SETTINGS_PROPERTIES;

   private GlobalSettings _settings = null;
   private GlobalSettings _defaultSettings = new GlobalSettings(
          "def-value1", "def-value2", SettingEnum.VAL1);
   private Properties _props = new Properties();
   private File _persistentSettings = new File(SETTINGS_FILE_PATH);;


   /**
    * Constructor also initializes setting values.
    */
   public GlobalServiceImpl() {
      initializeSettings();
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.globalview.GlobalService#getSettings()
    */
   @Override
   public synchronized GlobalSettings getSettings() {
      return _settings;
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.globalview.GlobalService#setSettings(com.vmware.samples.globalview.GlobalSettings)
    */
   @Override
   public synchronized void setSettings(GlobalSettings settings) throws Exception {
      // We keep the new settings in memory even if the data persistence fails
      _settings = settings;

      // Saving settings may throw a Java exception which will be converted to a JQuery implementation of AJAX
      saveSettings();
   }

   /**
    * Write settings out to properties file in the sample's data folder.
    * @throws Exception in case of error
    */
   private void saveSettings() throws Exception {
      _props.put("setting1.value", _settings.getSetting1());
      _props.put("setting2.value", _settings.getSetting2());
      // Save the enum internal name, not the toString value which is the display name.
      _props.put("setting3.value", _settings.getSetting3().name());

      // Create the properties file if settings are saved for the first time
      if (!_persistentSettings.exists()) {
         // Create the directory if necessary
         File dir = new File(SETTINGS_DIR_PATH);
         if (!dir.exists()) {
            if (!dir.mkdirs()) {
               String errorMsg = "Could not create directory " + SETTINGS_DIR_PATH +
                     ", check the write permission of the Virgo process.";
               _logger.error(errorMsg);
               throw new Exception(errorMsg);
            }
            _logger.info("Created directory " + dir.getPath());
         }

         try {
            _persistentSettings.createNewFile();
         } catch (IOException e) {
            // Log and re-throw the error to catch it in the UI
            String errorMsg = "Could not create file " + SETTINGS_FILE_PATH;
            _logger.error(errorMsg, e);
            throw new Exception(errorMsg, e);
         }
      }

      OutputStream out = null;
      try {
         out = new FileOutputStream(_persistentSettings);
         _props.store(out, "properties file for the globalview sample");
         _logger.info("Globalview settings saved in " + _persistentSettings.getPath());
      } catch (Exception e) {
         // Log and re-throw the error to catch it in the UI
         // (to test an error make settings.properties read-only after saving it once)
         String errorMsg = "Error while saving settings in " + SETTINGS_FILE_PATH;
         _logger.error(errorMsg, e);
         throw new Exception(errorMsg, e);
      } finally {
         if (out != null) {
            try {
               out.close();
            } catch (IOException e) {
               _logger.error(e);
            }
         }
      }
   }

   private void initializeSettings() {
      // Load the properties file holding the persistent setting values.
      // Try the persistent file first (saved from a previous session)
      InputStream in = null;

      try {
         in = new FileInputStream(_persistentSettings);
      } catch (FileNotFoundException e1) {
         // Not an error, no settings have been saved yet.
      }

      // Then try the settings.properties file stored with plugin bundle resources.
      if (in == null) {
         in = this.getClass().getClassLoader().getResourceAsStream(SETTINGS_PROPERTIES);
         if (in == null) {
            _logger.warn("Resource file missing: " + SETTINGS_PROPERTIES +
                  ", will use the hard-coded default settings instead");
            _settings = _defaultSettings;
            return;
         }
      }

      try {
         _props.load(in);
         String val1 = _props.getProperty("setting1.value");
         String val2 = _props.getProperty("setting2.value");
         SettingEnum val3 = SettingEnum.valueOf(_props.getProperty("setting3.value"));
         _settings = new GlobalSettings(val1, val2, val3);

      } catch (Exception e) {
         _logger.error(e);
         // use hard-coded settings in case of error
         _settings = _defaultSettings;
      } finally {
         if (in != null) {
            try {
               in.close();
            } catch (IOException e) {
               _logger.error(e);
            }
         }
      }
   }

   /**
    * Get a writable location where to save persistent data for this sample.
    * Here we use a folder "globalview" under the default data directory defined by
    * the env variable VMWARE_DATA_DIR + processName  (where processName will be either
    * vsphere-client for the Flex client, or vsphere-ui for the HTML client)
    * If the VMWARE_DATA_DIR is not set the "globalview" folder will be created in "server/temp/".
    * 
    * @return the path of the data folder.
    */
   private static String getGlobalViewDataFolder() {
      String dataFolder;
      String userName = System.getProperty("user.name");
      String baseDir =  System.getenv("VMWARE_DATA_DIR");
      dataFolder = (baseDir == null ? "temp" : baseDir ) + File.separator + userName + File.separator + "globalview";

      return dataFolder;
   }
}
