/* Copyright 2012 VMware, Inc. All rights reserved. -- VMware Confidential */
package com.vmware.samples.globalview.mvc;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import com.google.gson.Gson;
import com.vmware.samples.globalview.EchoService;
import com.vmware.samples.globalview.GlobalService;
import com.vmware.samples.globalview.model.GlobalSettings;

/**
 * A controller to serve HTTP JSON GET/POST requests to the endpoint "/services"
 * for the GlobalView sample
 */
@Controller
@RequestMapping(value = "/services")
public class ServicesController {

   private final EchoService _echoService;
   private final GlobalService _globalService;


   @Autowired
   public ServicesController(
         @Qualifier("echoService") EchoService echoService,
         @Qualifier("globalService") GlobalService globalService) {
      _echoService = echoService;
      _globalService = globalService;
   }

   /**
    * Echo a message back to the client.
    */
   @RequestMapping(value = "/echo", method = RequestMethod.POST)
   @ResponseBody
   public String echo(@RequestParam(value = "message", required = true) String message)
         throws Exception {
      return _echoService.echo(message);
   }


   /**
    * Get the current settings
    */
   @RequestMapping(value = "/getSettings", method = RequestMethod.GET)
   @ResponseBody
   public GlobalSettings getSettings() throws Exception {
      return _globalService.getSettings();
   }


   /**
    * Update settings
    */
   @RequestMapping(value = "/saveSettings", method = RequestMethod.POST)
   @ResponseBody
   public void setSettings(@RequestParam(value = "json", required = true) String json) throws Exception {
      Gson gson = new Gson();
      GlobalSettings settings = gson.fromJson(json, GlobalSettings.class);

      _globalService.setSettings(settings);
   }

   /**
    * Generic handling of internal exceptions.
    * Sends a 500 server error response along with a json body with messages
    *
    * @param ex The exception that was thrown.
    * @param response
    * @return a map containing the exception message, the cause, and a stackTrace
    */
   @ExceptionHandler(Exception.class)
   @ResponseBody
   public Map<String, String> handleException(Exception ex, HttpServletResponse response) {
      response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());

      Map<String,String> errorMap = new HashMap<String,String>();
      errorMap.put("message", ex.getMessage());
      if(ex.getCause() != null) {
         errorMap.put("cause", ex.getCause().getMessage());
      }
      StringWriter sw = new StringWriter();
      PrintWriter pw = new PrintWriter(sw);
      ex.printStackTrace(pw);
      errorMap.put("stackTrace", sw.toString());

      return errorMap;
   }
}

