/* Copyright 2016 VMware, Inc. All rights reserved. -- VMware Confidential */

package com.vmware.samples.vspherewssdk;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.vmware.samples.vspherewssdk.model.HostData;
import com.vmware.samples.vspherewssdk.model.VmData;
import com.vmware.samples.vspherewssdk.mvc.QueryUtil;
import com.vmware.vim25.DynamicProperty;
import com.vmware.vim25.HostConfigInfo;
import com.vmware.vim25.HostHardwareInfo;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.ObjectContent;
import com.vmware.vim25.ObjectSpec;
import com.vmware.vim25.PropertyFilterSpec;
import com.vmware.vim25.PropertySpec;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.SelectionSpec;
import com.vmware.vim25.ServiceContent;
import com.vmware.vim25.TraversalSpec;
import com.vmware.vim25.VimPortType;
import com.vmware.vim25.VimService;
import com.vmware.vim25.VirtualDevice;
import com.vmware.vim25.VirtualDisk;
import com.vmware.vim25.VirtualMachineConfigInfo;
import com.vmware.vise.data.query.DataService;
import com.vmware.vise.data.query.DataServiceExtensionRegistry;
import com.vmware.vise.data.query.PropertyProviderAdapter;
import com.vmware.vise.data.query.PropertyRequestSpec;
import com.vmware.vise.data.query.PropertyValue;
import com.vmware.vise.data.query.ResultItem;
import com.vmware.vise.data.query.ResultSet;
import com.vmware.vise.data.query.TypeInfo;
import com.vmware.vise.security.ClientSessionEndListener;
import com.vmware.vise.usersession.ServerInfo;
import com.vmware.vise.usersession.UserSession;
import com.vmware.vise.usersession.UserSessionService;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Example of a property provider using both DataService and the vSphere Web Services SDK (vim25)
 */
public class DataProviderImpl implements
      PropertyProviderAdapter, ClientSessionEndListener {

   private static final Log _logger = LogFactory.getLog(DataProviderImpl.class);
   private static final String VM_TYPE = "VirtualMachine";
   private static final String HOST_TYPE = "HostSystem";

   /*
    * Custom VM properties provided by this adapter. They must be qualified with their
    * own namespace to avoid collisions with the vSphere default space
    */
   private static final String VMDATA_PROPERTY = "samples:vmData";
   private static final String POWEREDON_PROPERTY = "samples:isPoweredOn";

   /*
    * Custom Host property provided by this adapter. It must be qualified with its
    * own namespace to avoid collisions with the vSphere default space
    */
   private static final String HOSTDATA_PROPERTY = "samples:hostData";

   private final UserSessionService _userSessionService;
   private final VimObjectReferenceService _vimObjectReferenceService;

   private static final String SERVICE_INSTANCE = "ServiceInstance";

   /** object for access to all of the methods defined in the vSphere API */
   private static VimPortType _vimPort = initializeVimPort();

   /** Reference to the DataService to make additional queries */
   private final DataService _dataService;

   private static VimPortType initializeVimPort() {
      // Static initialization is preferred because it takes a few seconds.
      VimService vimService = new VimService();
      return vimService.getVimPort();
   }

   /**
    * Static Initialization block, which will make this client trust all
    * certificates. USE THIS ONLY FOR TESTING.
    */
   static {
      HostnameVerifier hostNameVerifier = new HostnameVerifier() {
         @Override
         public boolean verify(String urlHostName, SSLSession session) {
            return true;
         }
      };
      HttpsURLConnection.setDefaultHostnameVerifier(hostNameVerifier);

      javax.net.ssl.TrustManager[] trustAllCerts = new javax.net.ssl.TrustManager[1];
      javax.net.ssl.TrustManager tm = new TrustAllTrustManager();
      trustAllCerts[0] = tm;
      javax.net.ssl.SSLContext sc = null;

      try {
         sc = javax.net.ssl.SSLContext.getInstance("SSL");
      } catch (NoSuchAlgorithmException e) {
         _logger.info(e);
      }

      if (sc != null) {
         javax.net.ssl.SSLSessionContext sslsc = sc.getServerSessionContext();
         sslsc.setSessionTimeout(0);
         try {
            sc.init(null, trustAllCerts, null);
         } catch (KeyManagementException e) {
            _logger.info(e);
         }
         javax.net.ssl.HttpsURLConnection.setDefaultSSLSocketFactory(
               sc.getSocketFactory());
      }
   }

   /**
    *  Constructor used to inject the utility services (see the declaration
    *  in main/resources/spring/bundle-context-osgi.xml)
    *
    * @param userSessionService
    *    Service to access the current session information.
    * @param vimObjectReferenceService
    *    Service to access vSphere object references information.
    * @param registry
    *    Registration interface for data adapters and providers.
    * @param dataService
    *    DataService used for various queries.
    */
   public DataProviderImpl(
            UserSessionService userSessionService,
            VimObjectReferenceService vimObjectReferenceService,
            DataServiceExtensionRegistry registry,
            DataService dataService) {
      _userSessionService = userSessionService;
      _vimObjectReferenceService = vimObjectReferenceService;
      _dataService = dataService;
      registry.registerDataAdapter(this, getProvidedTypeInfos());
   }

   /**
    * @return the types and properties handled by this adapter.
    */
   private TypeInfo[] getProvidedTypeInfos() {
      TypeInfo vmTypeInfo = new TypeInfo();
      vmTypeInfo.type = VM_TYPE;
      vmTypeInfo.properties = new String[] { VMDATA_PROPERTY, POWEREDON_PROPERTY };

      TypeInfo hostTypeInfo = new TypeInfo();
      hostTypeInfo.type = HOST_TYPE;
      hostTypeInfo.properties = new String[] { HOSTDATA_PROPERTY };

      return new TypeInfo[] { vmTypeInfo, hostTypeInfo };
   }

   /* (non-Javadoc)
    * @see com.vmware.vise.security.ClientSessionEndListener#sessionEnded(java.lang.String)
    */
   @Override
   public void sessionEnded(String clientId) {
      _logger.info("Logging out client session - " + clientId);
      // Clean up all session specific resources.
      // Logout from any session specific services.
   }

   /* (non-Javadoc)
    * @see com.vmware.vise.data.query.PropertyProviderAdapter#getProperties
    * (com.vmware.vise.data.query.PropertyRequestSpec)
    */
   @Override
   public ResultSet getProperties(PropertyRequestSpec propertyRequest) {
      // This is the main entry point of the PropertyProviderAdapter.
      // propertyRequest.properties contains the properties to be retrieved.

      // Note: validation of propertyRequest is not really necessary as this simple
      // PropertyProviderAdapter is only called for the property it was registered for.

      ResultSet result = new ResultSet();

      try {
         List<ResultItem> resultItems = new ArrayList<ResultItem>();

         // A PropertyProviderAdapter can handle multiple objects, although
         // the vsphere-wssdk-html client only uses it for one VM or Host at a time.
         for (Object objectRef: propertyRequest.objects) {
            ResultItem resultItem = null;
            String objectType =
                  _vimObjectReferenceService.getResourceObjectType(objectRef);
            if (objectType.equals(VM_TYPE)) {
               resultItem = getVmProperties(objectRef, propertyRequest);
            } else if (objectType.equals(HOST_TYPE)) {
               resultItem = getHostProperties(objectRef);
            }
            if (resultItem != null) {
               resultItems.add(resultItem);
            }
         }
         result.items = resultItems.toArray(new ResultItem[] {});

         // Note: result.totalMatchedObjectCount remains 0 in a PropertyProviderAdapter
         // since no new objects are discovered, we are just returning properties of
         // existing objects.

      } catch (Exception e) {
         _logger.error("DataProviderImpl.getProperties error: ", e);
         // Passing the exception in the result allows to display an error notification
         // in the client UI.
         result.error = e;
      }
      return result;
   }

   private ResultItem getVmProperties(Object vmRef, PropertyRequestSpec propertyRequest)
         throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg, Exception {
      ResultItem ri = new ResultItem();
      ri.resourceObject = vmRef;
      int numProperties = propertyRequest.properties.length;
      ri.properties = new PropertyValue[numProperties];

      for (int i = 0; i < numProperties; i++) {
         // Need full package because PropertySpec exists in both DataService API and vim25 API
         com.vmware.vise.data.PropertySpec propSpec = propertyRequest.properties[i];
         String property = propSpec.propertyNames[0];
         if (VMDATA_PROPERTY.equals(property)) {
            // Case of the VmData model used in the VM monitor view
            ri.properties[i] = getVmDataProperties(vmRef);
         }
         else if (POWEREDON_PROPERTY.equals(property)) {
            // Case of the isPoweredOn property used in the VM sub-menu
            ri.properties[i] = getPoweredOnProperty(vmRef);
         }
         else {
            // This should not happen since DataProvider was registered only for VMDATA_PROPERTY
            // and POWEREDON_PROPERTY
            _logger.error("Unhandled VM property! " + property);
         }
      }
      return ri;
   }

   /**
    * Handle the VM POWEREDON_PROPERTY using DataService because it's easier for simple properties
    *
    * @return a PropertyValue containing true if the VM is powered on, false otherwise
    */
   private PropertyValue getPoweredOnProperty(Object vmRef) throws Exception {

      // "runtime.powerState" is a native VM property whose value is a "VirtualMachine.PowerState" enum
      // We can't use the vim25 equivalent here but we can easily use toString() for the string value.
      String[] powerStateProp = new String[] { "runtime.powerState" };
      PropertyValue[] powerStateResult = QueryUtil.getProperties(_dataService, vmRef, powerStateProp);
      String powerState = powerStateResult[0].value.toString();

      // Transform result into a boolean value (required by <conditionalProperty> expression in plugin.xml)
      PropertyValue powerOnProp = new PropertyValue();
      powerOnProp.resourceObject = vmRef;
      powerOnProp.propertyName = POWEREDON_PROPERTY;
      powerOnProp.value = "poweredOn".equals(powerState) ? true : false;

      return powerOnProp;
   }

   /**
    * Get the properties defined in the VmData model, using the vSphere Web Services SDK
    *
    * @param vmRef
    *             The object reference passed by the client
    * @return The PropertyValue for that VM.
    *
    * @throws InvalidPropertyFaultMsg
    * @throws RuntimeFaultFaultMsg
    */
   private PropertyValue getVmDataProperties(Object vmRef)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {

      // Create a vim25 ManagedObjectReference for the VirtualMachine
      ManagedObjectReference vmMor = new ManagedObjectReference();
      vmMor.setType(_vimObjectReferenceService.getResourceObjectType(vmRef));
      vmMor.setValue(_vimObjectReferenceService.getValue(vmRef));

      // Initialize the ServiceContent interface.
      // TODO: this should be optimized further in case this class is going to be
      // used for multiple VMs/Hosts at a time, i.e. you can keep a map of serverGuid
      // to ServiceContent in order to avoid calling getServiceContent each time.
      String serverGuid = _vimObjectReferenceService.getServerGuid(vmRef);
      ServiceContent service = getServiceContent(serverGuid);
      if (service == null) {
         // We could also report an error here.
         return null;
      }

      VirtualMachineConfigInfo config = null;
      VmData vmData = new VmData();

      // Create Datacenter Property Spec
      com.vmware.vim25.PropertySpec propertySpec = new PropertySpec();
      propertySpec.setAll(Boolean.FALSE);
      propertySpec.setType("Datacenter");
      propertySpec.getPathSet().add("name");

      // Create VirtualMachine Property Spec
      com.vmware.vim25.PropertySpec propertySpec2 = new PropertySpec();
      propertySpec2.setAll(Boolean.FALSE);
      propertySpec2.setType(VM_TYPE);
      propertySpec2.getPathSet().add("config");

      // Create VirtualMachine Object Spec
      ObjectSpec objectSpec = new ObjectSpec();
      objectSpec.setObj(vmMor);
      objectSpec.setSkip(Boolean.FALSE);
      objectSpec.getSelectSet().addAll(buildTraversalSpecForVMToDatacenter());

      // Create PropertyFilterSpec using previous PropertySpec and ObjectSpec
      PropertyFilterSpec propertyFilterSpec = new PropertyFilterSpec();
      propertyFilterSpec.getPropSet().add(propertySpec);
      propertyFilterSpec.getPropSet().add(propertySpec2);
      propertyFilterSpec.getObjectSet().add(objectSpec);

      List<PropertyFilterSpec> propertyFilterSpecs = new ArrayList<PropertyFilterSpec>();
      propertyFilterSpecs.add(propertyFilterSpec);

      // Retrieve properties with the PropertyCollector
      List<ObjectContent> objectContents = _vimPort.retrieveProperties(
                  service.getPropertyCollector(), propertyFilterSpecs);
      if (objectContents != null) {
         for (ObjectContent content : objectContents) {
            List<DynamicProperty> dps = content.getPropSet();
            if (dps != null) {
               if (VM_TYPE.equalsIgnoreCase(content.getObj().getType())) {
                  for (DynamicProperty dp : dps) {
                     // Consider the first property only: the first VM found.
                     config = (VirtualMachineConfigInfo) dp.getVal();
                     break;
                  }
               } else {
                  for (DynamicProperty dp : dps) {
                     // Consider the first property only: datacenter name.
                     vmData.datacenterName = (String) dp.getVal();
                     break;
                  }
               }
            }
         }
      }
      long size = 0;
      if (config != null) {
         List<VirtualDevice> vDevices = config.getHardware().getDevice();
         for (VirtualDevice vDevice : vDevices) {
            if (vDevice instanceof VirtualDisk) {
               size += ((VirtualDisk) vDevice).getCapacityInKB();
            }
         }

         // Note: these strings should be localized in a real application
         vmData.capacityInKb = size + " KB";
         vmData.numberOfVirtualCpus = config.getHardware().getNumCPU() + " CPU";
         vmData.vmName = config.getName();
      }

      PropertyValue vmProp = new PropertyValue();
      vmProp.resourceObject = vmRef;
      vmProp.propertyName = VMDATA_PROPERTY;
      vmProp.value = vmData;

      return vmProp;
   }

   /**
    * Get the properties of an individual Host, in this case the ones
    * defined in the hostData model.
    *
    * @param hostRef
    *             The object reference passed by the client
    * @return ResultItem
    *             The result for that Host.
    *
    * @throws InvalidPropertyFaultMsg
    * @throws RuntimeFaultFaultMsg
    */
   private ResultItem getHostProperties(Object hostRef)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {

      // Create a vim25 ManagedObjectReference for the HostSystem
      ManagedObjectReference hostMor = new ManagedObjectReference();
      hostMor.setType(_vimObjectReferenceService.getResourceObjectType(hostRef));
      hostMor.setValue(_vimObjectReferenceService.getValue(hostRef));

      String serverGuid = _vimObjectReferenceService.getServerGuid(hostRef);
      ServiceContent service = getServiceContent(serverGuid);
      if (service == null) {
         // We could also report an error here.
         return null;
      }

      com.vmware.vim25.PropertySpec propertySpec = new com.vmware.vim25.PropertySpec();
      propertySpec.setAll(Boolean.FALSE);
      propertySpec.setType(HOST_TYPE);
      propertySpec.getPathSet().add("name");

      com.vmware.vim25.PropertySpec propertySpec2 = new com.vmware.vim25.PropertySpec();
      propertySpec2.setAll(Boolean.FALSE);
      propertySpec2.setType(HOST_TYPE);
      propertySpec2.getPathSet().add("config");

      com.vmware.vim25.PropertySpec propertySpec3 = new com.vmware.vim25.PropertySpec();
      propertySpec3.setAll(Boolean.FALSE);
      propertySpec3.setType(HOST_TYPE);
      propertySpec3.getPathSet().add("hardware");

      // Create VirtualMachine Property Spec
      com.vmware.vim25.PropertySpec propertySpec4 = new com.vmware.vim25.PropertySpec();
      propertySpec4.setAll(Boolean.FALSE);
      propertySpec4.setType(VM_TYPE);
      propertySpec4.getPathSet().add("config");

      // Create Host Object Spec
      ObjectSpec objectSpec = new ObjectSpec();
      objectSpec.setObj(hostMor);
      objectSpec.setSkip(Boolean.FALSE);

      TraversalSpec tSpec = new TraversalSpec();
      tSpec.setName("traverseEntities");
      tSpec.setPath("view");
      tSpec.setSkip(false);
      tSpec.setType("ContainerView");
      objectSpec.getSelectSet().add(tSpec);

      // HostSystem -> VM
      TraversalSpec hToVm = new TraversalSpec();
      hToVm.setType("HostSystem");
      hToVm.setPath("vm");
      hToVm.setName("hToVm");
      hToVm.setSkip(Boolean.FALSE);
      objectSpec.getSelectSet().add(hToVm);

      // Create PropertyFilterSpec using previous PropertySpec and ObjectSpec
      PropertyFilterSpec propertyFilterSpec = new PropertyFilterSpec();
      propertyFilterSpec.getPropSet().add(propertySpec);
      propertyFilterSpec.getPropSet().add(propertySpec2);
      propertyFilterSpec.getPropSet().add(propertySpec3);
      propertyFilterSpec.getPropSet().add(propertySpec4);
      propertyFilterSpec.getObjectSet().add(objectSpec);

      List<PropertyFilterSpec> propertyFilterSpecs = new ArrayList<PropertyFilterSpec>();
      propertyFilterSpecs.add(propertyFilterSpec);

      // Retrieve properties with the PropertyCollector
      VirtualMachineConfigInfo vmConfig = null;
      HostHardwareInfo hardwareInfo = null;
      HostConfigInfo hostConfig = null;
      String hostName = null;
      List<ObjectContent> objectContents = _vimPort.retrieveProperties(
                  service.getPropertyCollector(), propertyFilterSpecs);
      if (objectContents != null) {
         for (ObjectContent content : objectContents) {
            List<DynamicProperty> dps = content.getPropSet();
            if (dps != null) {
               if (VM_TYPE.equalsIgnoreCase(content.getObj().getType()) && vmConfig == null) {
                  for (DynamicProperty dp : dps) {
                     // First VM found
                     vmConfig = (VirtualMachineConfigInfo) dp.getVal();
                     break;
                  }
               } else if (HOST_TYPE.equalsIgnoreCase(content.getObj().getType())) {
                  for (DynamicProperty dp : dps) {
                     if(dp.getName().equalsIgnoreCase("name")){
                        hostName = (String) dp.getVal();
                     } else if(dp.getName().equalsIgnoreCase("config")){
                        hostConfig = (HostConfigInfo) dp.getVal();
                     } else {
                        hardwareInfo = (HostHardwareInfo) dp.getVal();
                     }
                  }
               }
            }
         }
      }

      VmData vmData = new VmData();
      long size = 0;
      if (vmConfig != null) {
         List<VirtualDevice> vDevices = vmConfig.getHardware().getDevice();
         for (VirtualDevice vDevice : vDevices) {
            if (vDevice instanceof VirtualDisk) {
               size += ((VirtualDisk) vDevice).getCapacityInKB();
            }
         }
         vmData.capacityInKb = size + " KB";
         vmData.numberOfVirtualCpus = vmConfig.getHardware().getNumCPU() + " CPU";
         vmData.vmName = vmConfig.getName();
      }

      HostData hostData = new HostData();
      if (hostConfig != null) {
         hostData.name  = hostConfig.getNetwork().getDnsConfig().getHostName();
      }
      if (hardwareInfo != null) {
         hostData.uuid = hardwareInfo.getSystemInfo().getUuid();
      }
      hostData.ipAddress = hostName;

      ResultItem ri = new ResultItem();
      ri.resourceObject = hostRef;
      PropertyValue hostProp = new PropertyValue();
      hostProp.resourceObject = hostRef;
      hostProp.propertyName = HOSTDATA_PROPERTY;
      hostProp.value = hostData;
      PropertyValue vmProp = new PropertyValue();
      vmProp.resourceObject = hostRef;
      vmProp.propertyName = VMDATA_PROPERTY;
      vmProp.value = vmData;
      ri.properties = new PropertyValue[] { hostProp, vmProp };

      return ri;
   }

   /**
    *
    * @return An array of SelectionSpec to navigate from the VM and move
    *         upwards to reach the Datacenter
    */
   private static List<SelectionSpec> buildTraversalSpecForVMToDatacenter() {

      // For Folder -> Folder recursion
      SelectionSpec sspecvfolders = new SelectionSpec();
      sspecvfolders.setName("VisitFolders");

      TraversalSpec visitFolders = new TraversalSpec();
      visitFolders.setType("Folder");
      visitFolders.setPath("parent");
      visitFolders.setSkip(Boolean.FALSE);
      visitFolders.setName("VisitFolders");
      visitFolders.getSelectSet().add(sspecvfolders);

      // For vApp -> vApp recursion
      SelectionSpec sspecvApp = new SelectionSpec();
      sspecvApp.setName("vAppToVApp");

      SelectionSpec sspecvAppToFolder = new SelectionSpec();
      sspecvAppToFolder.setName("vAppToFolder");

      // For vApp -> parent folder
      TraversalSpec vAppToFolder = new TraversalSpec();
      vAppToFolder.setType("VirtualApp");
      vAppToFolder.setPath("parentFolder");
      vAppToFolder.setSkip(Boolean.FALSE);
      vAppToFolder.setName("vAppToFolder");
      vAppToFolder.getSelectSet().add(sspecvfolders);

      // For vApp -> parent vApp
      TraversalSpec vAppToVApp = new TraversalSpec();
      vAppToVApp.setType("VirtualApp");
      vAppToVApp.setPath("parentVApp");
      vAppToVApp.setSkip(Boolean.FALSE);
      vAppToVApp.setName("vAppToVApp");
      vAppToVApp.getSelectSet().add(sspecvApp);
      vAppToVApp.getSelectSet().add(sspecvAppToFolder);

      // For VM -> parent vApp
      TraversalSpec vmTovApp = new TraversalSpec();
      vmTovApp.setType(VM_TYPE);
      vmTovApp.setPath("parentVApp");
      vmTovApp.setSkip(Boolean.FALSE);
      vmTovApp.setName("vmTovApp");
      vmTovApp.getSelectSet().add(vAppToVApp);
      vmTovApp.getSelectSet().add(vAppToFolder);

      // For VM -> parent folder
      TraversalSpec vmToFolder = new TraversalSpec();
      vmToFolder.setType(VM_TYPE);
      vmToFolder.setPath("parent");
      vmToFolder.setSkip(Boolean.FALSE);
      vmToFolder.setName("vmToFolder");
      vmToFolder.getSelectSet().add(sspecvfolders);

      List<SelectionSpec> speclist = new ArrayList<SelectionSpec>();
      speclist.add(vmToFolder);
      speclist.add(vmTovApp);
      speclist.add(visitFolders);
      return speclist;
   }

   /**
    * Find the relevant ServerInfo Object for the given serverGuid and the
    * current session.
    */
   private ServerInfo getServerInfoObject(String serverGuid) {
      UserSession userSession = _userSessionService.getUserSession();

      for (ServerInfo sinfo : userSession.serversInfo) {
         if (sinfo.serviceGuid.equalsIgnoreCase(serverGuid)) {
            return sinfo;
         }
      }
      return null;
   }


   /**
    * Get the ServiceContent for a given VC server.
    */
   private ServiceContent getServiceContent(String serverGuid) {

      ServerInfo serverInfoObject = getServerInfoObject(serverGuid);
      String sessionCookie = serverInfoObject.sessionCookie;
      String serviceUrl = serverInfoObject.serviceUrl;

      if (_logger.isDebugEnabled()) {
         _logger.debug("getServiceContent: sessionCookie = "+ sessionCookie +
                        ", serviceUrl= " + serviceUrl);
      }

      List<String> values = new ArrayList<String>();
      values.add("vmware_soap_session=" + sessionCookie);
      Map<String, List<String>> reqHeadrs = new HashMap<String, List<String>>();
      reqHeadrs.put("Cookie", values);

      Map<String, Object> reqContext = ((BindingProvider)_vimPort).getRequestContext();
      reqContext.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, serviceUrl);
      reqContext.put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);
      reqContext.put(MessageContext.HTTP_REQUEST_HEADERS, reqHeadrs);

      final ManagedObjectReference svcInstanceRef = new ManagedObjectReference();
      svcInstanceRef.setType(SERVICE_INSTANCE);
      svcInstanceRef.setValue(SERVICE_INSTANCE);

      ServiceContent serviceContent = null;
      try {
         serviceContent = _vimPort.retrieveServiceContent(svcInstanceRef);
      } catch (RuntimeFaultFaultMsg e) {
         _logger.error("getServiceContent error: " + e);
      }

      return serviceContent;
   }

   /**
    * The following code snippet will trust all certificates Used only in
    * testing.
    */
   private static class TrustAllTrustManager implements
      javax.net.ssl.TrustManager, javax.net.ssl.X509TrustManager {

      @Override
      public java.security.cert.X509Certificate[] getAcceptedIssuers() {
         return null;
      }

      @Override
      public void checkServerTrusted(java.security.cert.X509Certificate[] certs,
            String authType) throws java.security.cert.CertificateException {
         return;
      }

      @Override
      public void checkClientTrusted(java.security.cert.X509Certificate[] certs,
            String authType) throws java.security.cert.CertificateException {
         return;
      }
   } // End class TrustAllTrustManager.

} // End class VmPropertyProviderImpl.
