/** Copyright 2013 VMware, Inc. All rights reserved. -- VMware Confidential */
package com.vmware.samples.vspherewssdk.mvc;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.gson.Gson;
import com.vmware.samples.vspherewssdk.VmActionService;
import com.vmware.samples.vspherewssdk.model.VmInfo;
import com.vmware.vise.data.query.ObjectReferenceService;

/**
 * A controller to serve HTTP JSON GET/POST requests to the endpoint "/actions.html".
 */
@Controller
@RequestMapping(value = "/actions.html")
public class ActionsController {
   private final static Log _logger = LogFactory.getLog(ActionsController.class);

   // UI plugin resource bundle for localized messages
   private final String RESOURCE_BUNDLE = "com_vmware_samples_vspherewssdk";

   private final VmActionService _vmActionService;
   private final ObjectReferenceService _objectReferenceService;

   @Autowired
   public ActionsController(
         VmActionService vmActionService,
         @Qualifier("objectReferenceService") ObjectReferenceService objectReferenceService) {
      _vmActionService = vmActionService;
      _objectReferenceService = objectReferenceService;
      QueryUtil.setObjectReferenceService(objectReferenceService);
   }

   /**
    * POST method endpoint for actions on VM objects in this sample.
    *
    * @param actionUid  the action Uid as defined in plugin.xml
    *
    * @param targets  comma-separated list of object ids
    *
    * @param json additional data in JSON format, or null.
    *
    * @return an actionResult Json map.
    */
   @RequestMapping(method = RequestMethod.POST)
   @ResponseBody
   public Map<String, Object> invoke(
            @RequestParam(value = "actionUid", required = true) String actionUid,
            @RequestParam(value = "targets", required = true) String targets,
            @RequestParam(value = "json", required = false) String json)
            throws Exception {

      ActionResult actionResult = new ActionResult(actionUid, RESOURCE_BUNDLE);

      // Parameters validation
      Object vmRef = null;
      String[] objectIds = targets.split(",");
      if (objectIds.length > 1) {
         // Our VM actions only support 1 target object
         _logger.warn("Action " + actionUid + " called with " + objectIds.length
               + " target objects, will use only the first one");
      }

      // Find the VM object reference
      String objectId = ObjectIdUtil.decodeParameter(objectIds[0]);
      vmRef = _objectReferenceService.getReference(objectId);

      if (vmRef == null) {
         // Corner case where the VM was already deleted
         String errorMsg = "VM not found with id: " + objectId;
         _logger.error(errorMsg);
         actionResult.setErrorLocalizedMessage(errorMsg);
         return actionResult.getJsonMap();
      }

      VmInfo vmInfo = null;
      if (json != null) {
         // Create a VmInfo parameter from the json data.
         Gson gson = new Gson();
         vmInfo = gson.fromJson(json, VmInfo.class);
      }

      if (actionUid.equals("com.vmware.samples.vspherewssdk.myVmAction1")) {
         // Action1 using a vmInfo parameter
         boolean result = _vmActionService.backendAction1(vmRef, vmInfo);
         actionResult.setObjectChangedResult(result, "action1.errorMsg");

      } else if (actionUid.equals("com.vmware.samples.vspherewssdk.myVmAction2")) {
         // Action2 is headless and without parameters
         boolean result = _vmActionService.backendAction2(vmRef);
         actionResult.setObjectChangedResult(result, "action2.errorMsg");

      } else {
         String warning = "Action not implemented yet! "+ actionUid;
         _logger.warn(warning);
         actionResult.setErrorLocalizedMessage(warning);
      }
      return actionResult.getJsonMap();
   }

   /**
    * Generic handling of internal exceptions.
    * Sends a 500 server error response along with a json body with messages
    *
    * @param ex The exception that was thrown.
    * @param response
    * @return a map containing the exception message, the cause, and a stackTrace
    */
   @ExceptionHandler(Exception.class)
   @ResponseBody
   public Map<String, String> handleException(Exception ex, HttpServletResponse response) {
      response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());

      Map<String,String> errorMap = new HashMap<String,String>();
      errorMap.put("message", ex.getMessage());
      if(ex.getCause() != null) {
         errorMap.put("cause", ex.getCause().getMessage());
      }
      StringWriter sw = new StringWriter();
      PrintWriter pw = new PrintWriter(sw);
      ex.printStackTrace(pw);
      errorMap.put("stackTrace", sw.toString());

      return errorMap;
   }
}

