-- Used for UM DB migration from 3.5 to 3.6

-- Attention: this file should be synced with postgresql-create.sql:
-- when a change is made here, the other should be also updated. The final schemas should be the same.

CREATE TABLE "KeyValueStore"(
   "key" varchar(255) NOT NULL PRIMARY KEY,
   "value" varchar(512) NOT NULL,
   "createDate" timestamp NOT NULL,
   "changedDate" timestamp,
   "note" varchar(255)
);

CREATE TABLE "KeyValueStoreHistory"(
   "id" bigint NOT NULL PRIMARY KEY,
   "key" varchar(255) NOT NULL, 
   "oldValue" varchar(512),
   "newValue" varchar(512),   
   "createDate" timestamp NOT NULL,
   "note" varchar(255)
);

CREATE SEQUENCE "s_KeyValueStoreHistory_id"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

alter table "Cluster" rename to "ClusterOld";
create table "Cluster" (
    "id"   integer primary key not null,
    "clusName" varchar(128) not null,
    "dataStoreName" varchar(128) not null,
    "vcServerId" integer not null,
    "vsanEnabled" boolean not null,
    "lastChanged" timestamp not null
);

insert into "Cluster"("id", "clusName", "dataStoreName", "vcServerId", "vsanEnabled", "lastChanged")
select "id", "name", 'unknownyet', "vcServerId", "vsanEnabled", "lastChanged" from "ClusterOld";

-- create sequence for cluster table after data is migrated.
CREATE OR REPLACE FUNCTION create_cluster_sequence()
RETURNS integer AS $$
DECLARE
  cntOfRows   integer;
  nextId      integer;
BEGIN
   select count(*) into cntOfRows from "Cluster";
   if (cntOfRows = 0) then
     nextId = 1;
   else
     select max(id)+1 into nextId from "Cluster";
   end if;

   return nextId;
END
$$ LANGUAGE plpgsql;
select create_cluster_sequence() as startvalue \gset
CREATE SEQUENCE "s_Cluster_id" START WITH :startvalue;

drop table "ClusterOld";

alter table "ClusterVsanHistory" rename to "ClusterVsanHistoryBkup";
drop sequence "s_ClusterVsanHistory_id";
create table "ClusterVsanHistory" (
    "id" integer primary key not null,
    "vcServerId" integer not null,
    "clusterId" integer not null,
    "vsanLicCategory" integer not null,
    "usedTotalMb" bigint not null,
    "deDupEnabled" boolean not null default(false),
    "erasureCoding" boolean not null default(false),
    "stretchedClusterEnabled" boolean not null default(false),
    "iopsLimitUsed" boolean not null default(false),
    "startTime" timestamp not null,
    "endTime" timestamp not null
  );
create sequence "s_ClusterVsanHistory_id";

CREATE TYPE vcclusters AS
("vcServerId" integer, "clusterName" varchar, "firstId" bigint);

CREATE OR REPLACE FUNCTION get_clusters()
RETURNS SETOF vcclusters AS
'select "vcServerId", "clusterName", min(id) as "firstId" from "ClusterVsanHistoryBkup" group by "vcServerId", "clusterName";'
language sql;

alter table "ClusterVsanHistoryBkup" add column "endTime" timestamp;
alter table "ClusterVsanHistoryBkup" add column "checkTime" timestamp;
update "ClusterVsanHistoryBkup" set "endTime" = "collectTime"+interval '1 hour';
update "ClusterVsanHistoryBkup" set "checkTime" = "collectTime"+interval '1 day';

CREATE TYPE vsanhist AS
("vcServerId" integer, "clusterName" varchar, "vsanLicCategory" integer,
 "usedTotalMb" bigint, "startTime" timestamp, "endTime" timestamp);

CREATE OR REPLACE FUNCTION migrate_vsan_history(currMigrationGraceTime interval, farFuture timestamp)
     RETURNS SETOF vsanhist AS
$BODY$
DECLARE
    r    vcclusters%rowtype;
    license integer;
    usedTotal bigint;
    startT  timestamp;
    endT    timestamp;
    checkT  timestamp;
    cur_vsan refcursor;
    rec   record;
BEGIN
    for r in select * from get_clusters() loop
       select "vsanLicCategory", "usedTotalMb", "collectTime", "endTime", "checkTime"
       into license, usedTotal, startT, endT, checkT
       from "ClusterVsanHistoryBkup" v where v.id = r."firstId" and v."vcServerId" = r."vcServerId"
       and v."clusterName" = r."clusterName";

       open cur_vsan for select "vsanLicCategory", "usedTotalMb", "collectTime", "endTime", "checkTime"
           from "ClusterVsanHistoryBkup"
           where "vcServerId" = r."vcServerId" and "clusterName" = r."clusterName"
           and "id" > r."firstId";
       loop
         fetch cur_vsan into rec;
         exit when not found;

         if (rec."vsanLicCategory" != license OR rec."usedTotalMb" != usedTotal OR
             rec."collectTime" > checkT) then
             if (rec."collectTime" > checkT) then
               return next (r."vcServerId", r."clusterName", license, usedTotal, startT, endT+interval '23 hours');
             else
               return next (r."vcServerId", r."clusterName", license, usedTotal, startT, endT);
             end if;

           startT := rec."collectTime";
           license := rec."vsanLicCategory";
           usedTotal := rec."usedTotalMb";
         end if;
         endT   := rec."endTime";
         checkT := rec."checkTime";

       end loop;
       if (now() - endT < currMigrationGraceTime) then
         return next (r."vcServerId", r."clusterName", license, usedTotal, startT, farFuture);
       else
         return next (r."vcServerId", r."clusterName", license, usedTotal, startT, endT);
       end if;

       close cur_vsan;
    end loop;
    return;
END
$BODY$
language plpgsql;

-- Here we take that if the last collection time in 3.5 is within 6 hours of now, VSAN
-- is alive in 3.6 so we keep the endTime open; otherwise we are not sure if VSAN is alive
-- so we will have the last metering interval closed with the last collection time.
insert into "ClusterVsanHistory" ("id", "vcServerId", "clusterId", "vsanLicCategory",
                                  "usedTotalMb", "startTime", "endTime")
select nextval('"s_ClusterVsanHistory_id"'), v."vcServerId", c."id", v."vsanLicCategory", v."usedTotalMb",
       v."startTime", v."endTime"
from migrate_vsan_history(interval '6 hours', timestamp '3000-01-01 00:00:00.000') v, "Cluster" c
where c."vcServerId" = v."vcServerId"
and c."clusName" = v."clusterName";

--------  Migration "HostHistory"."nsx" values to "VmHistory"."nsx"  ---------------

alter table "Vm" add column "nsxUsageLicense" integer not null default(0);

alter table "VmHistory" add column "nsxUsageLicense" integer not null default(0);

alter table "NsxServer" drop column if exists "licenseType";

-- tmp_update_vmh_id will contain vm history records that orevelaped with host hostory.
--   those records must be splitted to remove overlapping.
-- Note: This table will be relativly small because HostHistory table doesn't chnage too much while VM are online.
create table "tmp_update_vmh_id" (
    "vmh_id"   integer,
    "startTime" timestamp,
    "endTime" timestamp,
    "nsx" integer NOT NULL,
    "vmId" integer NOT NULL,
    "hostId" integer NOT NULL,
    "connState" integer NOT NULL,
    "powerState" integer NOT NULL,
    "vmMemSize" integer NOT NULL,
    "vmMemRes" integer NOT NULL,
    "guestFullName" character varying(1000),
    "cpuCount" integer NOT NULL,
    "vrops" boolean NOT NULL
);

insert into "tmp_update_vmh_id"("vmh_id", "startTime", "endTime", "nsx", "vmId", "hostId","connState","powerState","vmMemSize",
    "vmMemRes","guestFullName","cpuCount","vrops")
select
       vh.id                 "vmhId",
       greatest(vh.time, hh.time)                         "intervalStart",
       least(vh."nextTime", hh."nextTime")                "intervalEnd",
       hh."nsx",
       vh."vmId", vh."hostId",vh."connState",vh."powerState",vh."vmMemSize",
       vh."vmMemRes",vh."guestFullName",vh."cpuCount",vh."vrops"
FROM "VmHistory" vh, "HostHistory" hh
WHERE vh."hostId" = hh."hostId"
AND (vh.time != greatest(vh.time, hh.time) OR vh."nextTime" != least(vh."nextTime", hh."nextTime"))
AND (vh.time, vh."nextTime") overlaps (hh.time, hh."nextTime");

-- Since duplicate rows are in tmp_update_vmh_id, we can clean up VmHistory.
--  Clean up first required because if restrictions that VmHistory table has.

DELETE FROM "VmHistory" vh
USING "tmp_update_vmh_id" tmp
WHERE tmp.vmh_id = vh.id;

-- Copy all records from tmp_update_vmh_id to VmHistory
insert into "VmHistory" (id, "vmId", "time", "nextTime", "hostId", "connState", "powerState", "vmMemSize", "vmMemRes", "guestFullName", "cpuCount","vrops")
select
       nextval('"s_VmHistory_id"'), tmp."vmId", tmp."startTime", tmp."endTime", tmp."hostId", tmp."connState", tmp."powerState",
       tmp."vmMemSize", tmp."vmMemRes", tmp."guestFullName", tmp."cpuCount", tmp."vrops"
FROM
      tmp_update_vmh_id tmp;

drop table "tmp_update_vmh_id";

-- Now we need to update nsxUsageLicense values at VmHistory & Vm tables.
-- At this point VmHistory is not overlapped with HostHistory. So VmHistory mapped 1 to 1 with HostHistory
UPDATE  "VmHistory" AS vh
SET
    "nsxUsageLicense" = hh."nsx"
FROM
    "HostHistory" hh
WHERE
    vh."hostId" = hh."hostId"
    AND (vh.time, vh."nextTime") overlaps (hh.time, hh."nextTime");
-- Update of Vm table is nice to have. In any case this data will be updated after next collection.
--  Vm will get the latest values form VmHistory
UPDATE  "Vm" AS v
SET
    "nsxUsageLicense" = vh."nsxUsageLicense"
FROM
    "VmHistory" vh
WHERE
    vh."vmId" = v.id
    AND vh."nextTime" > now();

-- DONE with nsx data migration!  Now we can delete the old data form HostHistory

-- HostHistory.nsx involve on many functions and views. So we need to update dependent first and remove HostHistory.nsx next
CREATE OR REPLACE FUNCTION "vm_history"(report_starttime timestamp, report_endtime timestamp, license_ids int[])
    RETURNS SETOF "VmHistory4Report"
AS $BODY$
SELECT v.id                  "vmId",
       v."vcServerId"        "vcId",
       v."vcName",
       v."guestHostName",
       v."instanceUuid",
       v."moref",
       v."reportExclusionReason",
       vc."host"             "vcIP",
       vh.id                 "vmhId",
       vh."hostId",
       vh."connState",
       vh."powerState",
       vh.time               "vhTime",
       vh."nextTime"         "vhNextTime",
       vh."vmMemSize",
       vh."vmMemRes",
       vh."cpuCount",
       vh."vrops",
       hh."name"             "hhName",
       hh."memSize"          "hhMemSize",
       hh."licenseId",
       hh.time               "hhTime",
       hh."nextTime"         "hhNextTime",
       vh."nsxUsageLicense",
       hh."vcd",
       hh."licenseCategoryId",
       l."name"               "licenseName",
       greatest(vh.time, hh.time)                         "intervalStart",
       least(vh."nextTime", hh."nextTime")                "intervalEnd",
       greatest(vh."vmMemSize"/2, vh."vmMemRes")          "billingMem"
FROM "Vm" v, "VmHistory" vh, "HostHistory" hh, "License" l, "VcServer" vc
WHERE v.id = vh."vmId" AND v."vcServerId" = vc.id AND vh."hostId" = hh."hostId"
AND hh."licenseId" = l.id AND hh."licenseCategoryId" = 1
AND (vh.time, vh."nextTime") overlaps (report_starttime, report_endtime)
AND (hh.time, hh."nextTime") overlaps (report_starttime, report_endtime)
AND (vh.time, vh."nextTime") overlaps (hh.time, hh."nextTime")
AND vh."connState" = 1 AND vh."powerState" = 1
AND ((array_length(license_ids, 1) is null) or (hh."licenseId" = any (license_ids)))
AND v."reportExclusionReason" is null
$BODY$
      LANGUAGE sql;

-- Drop and recreate views in order to delete "nsx" @ "HostHistory"
DROP VIEW "LicenseHostCount";
DROP VIEW "HostLatest";
DROP VIEW  "HostLatestHistory";
alter table "HostHistory" drop column if exists "nsx";

create view "HostLatestHistory" as
    select * from "HostHistory" a where exists(
        select * from (
            select distinct "hostId", max(time) over (partition by "hostId") latest from "HostHistory"
        ) b where a.time = b.latest and a."hostId" = b."hostId"
    );

CREATE VIEW "HostLatest" AS
    SELECT
        h.id,
        hl.id "hhId",
        h.uuid,
        h."vcServerId",
        moref,
        time,
        "nextTime",
        "fullName",
        version,
        name,
        "licenseId",
        "memSize"
    FROM "Host" h
        JOIN "HostLatestHistory" hl ON h.id = hl."hostId";

create view "LicenseHostCount" as
    select l.id, l.name, l.code, count(l.code) as "numHosts" from "License" l, "HostLatestHistory" h
        where h."licenseId" = l.id
        group by l.id, l.name, l.code;
-- HostHistory.nsx  is deleted and all dependents are recteated

--------  Done with Migration "HostHistory"."nsx" values to "VmHistory"."nsx"  ---------------


CREATE TYPE "VsanSpaceUsage" AS
("vcServerId" integer, "clusterId" integer, "vsanLicCategory" integer,
 "usedTotalMb" bigint, "intervalSeconds" double precision);

CREATE OR REPLACE FUNCTION "vsan_space_usage_std_license"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 1
      union
      SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = false AND "erasureCoding" = false) AND
      ("stretchedClusterEnabled" = false AND "iopsLimitUsed" = false)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_adv_license"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 2
      union
      SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = true OR "erasureCoding" = true) AND
      ("stretchedClusterEnabled" = false AND "iopsLimitUsed" = false)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_std_with_addon"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = false AND "erasureCoding" = false) AND
      ("stretchedClusterEnabled" = true OR "iopsLimitUsed" = true)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_adv_with_addon"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = true OR "erasureCoding" = true) AND
      ("stretchedClusterEnabled" = true OR "iopsLimitUsed" = true)) v
$BODY$
    LANGUAGE sql;

CREATE TYPE "VsanClusterInterval" AS
("vcServerId" integer, "clusterId" integer, "vsanLicCategory" integer, "usedTotalMb" bigint,
 "startTime" timestamp, "endTime" timestamp, "deDupEnabled" boolean, "erasureCoding" boolean,
 "stretchedClusterEnabled" boolean, "iopsLimitUsed" boolean
);

CREATE OR REPLACE FUNCTION "vsan_cluster_history"(timestamp, timestamp)
    RETURNS SETOF "VsanClusterInterval" AS $BODY$
SELECT "vcServerId", "clusterId", "vsanLicCategory", "usedTotalMb", "startTime", "endTime",
       "deDupEnabled", "erasureCoding", "stretchedClusterEnabled", "iopsLimitUsed"
FROM   "ClusterVsanHistory"
WHERE  ("startTime", "endTime") overlaps ($1, $2)
$BODY$
    LANGUAGE sql;

create table "CustomerCluster" (
    "id" integer primary key not null,
    "customerId" integer not null,
    "clusterId" integer not null,
    "customerVmRuleId" integer not null,
    "changed" timestamp not null
  );
create sequence "s_CustomerCluster_id";
create unique index "customeCluster_clusterId" on "CustomerCluster" ("clusterId");

---------vrops data migration, need to delete those line where vm is powered off at the time.------
ALTER TABLE "VcopsVmCollection" RENAME TO "VcopsVmCollectionOld";
CREATE TABLE "VcopsVmCollection" AS
SELECT distinct vvc.*
       FROM "VcopsVmCollectionOld" vvc
       INNER JOIN "VmHistory" vh ON vvc."vmId" = vh."vmId"
       where vvc.timestamp >= vh.time and vvc.timestamp < vh."nextTime" and vh."powerState" =1 and vh."connState"=1
;
DROP TABLE "VcopsVmCollectionOld";
alter table "VcopsVmCollection" add primary key ("id");
-- indexes on VcopsVmCollection
create index "VcopsVmCollection_timestamp" on "VcopsVmCollection" ("timestamp");
alter table "VcopsVmCollection" add constraint "VcopsVmCollectionFK23" foreign key ("vmId") references "Vm"("id");
alter table "VcopsVmCollection" add constraint "VcopsVmCollectionFK24" foreign key ("vcopsLicenseNameId") references "VcopsLicenseName"("id");
alter table "VcopsVmCollection" alter column "vmId" set not null;
alter table "VcopsVmCollection" alter column "timestamp" set not null;
alter table "VcopsVmCollection" alter column "vcopsLicenseNameId" set not null;
---------End of vrops data migration------

--meter unmanaged vcs in vrops
CREATE TABLE "UnManagedVcVrops" (
    "id" integer primary key not null,
    "vcUuid" varchar(255) not null,
    "vropsServerId" integer not null,
    "timestamp" timestamp not null,
    "vmCount" integer not null
  );
CREATE SEQUENCE "s_UnManagedVcVrops_id";
alter table "UnManagedVcVrops" add constraint "UnManagedVcVropsFK1" foreign key ("vropsServerId") references "VcopsServer"("id");

--meter DR in vcd
CREATE TABLE "VcdDR" (
    "id" integer primary key not null,
    "vcdId" integer not null,
    "orgName" varchar(255) not null,
    "orgId" varchar(255) not null,
    "numVms" integer not null,
    "collectTime" timestamp not null
  );
CREATE SEQUENCE "s_VcdDR_id";
alter table "VcdDR" add constraint "VcdDRFK1" foreign key ("vcdId") references "VcdServer"("id");

-- Alter Vm table for DFW vNIC and MAC support
alter table "Vm" add column "mac" varchar(128);

-- According https://bugzilla.eng.vmware.com/show_bug.cgi?id=1792573  we have at least one customers for whom open
--   interval is not '3000-01-01'
-- Note: psql treat Date&Time strings as UTC unless timezone is specified.
UPDATE "VmHistory" SET "nextTime"='3000-01-01' WHERE "nextTime">'2999-01-01';
UPDATE "HostHistory" SET "nextTime"='3000-01-01' WHERE "nextTime">'2999-01-01';


-- Mark migration is done.
UPDATE "SchemaInfo" SET "version" = 16, "migrationDone" = true;

--For support PSC on separate VM
CREATE SEQUENCE "s_PscServer_id";
CREATE TABLE "PscServer" (
"id"   integer PRIMARY KEY NOT NULL,
"host" varchar(128) NOT NULL,
"port" integer DEFAULT 7444 NOT NULL,
"createTime" timestamp NOT NULL
);
ALTER TABLE "VcServer" ADD COLUMN "sso" integer DEFAULT 0;
ALTER TABLE "VcServer" ADD COLUMN "pscId" integer;
ALTER TABLE "VcServer" ADD CONSTRAINT "VcServerFK01" FOREIGN KEY ("pscId") REFERENCES "PscServer"("id");
ALTER TABLE "VcServer" ADD CONSTRAINT "VcServerCHK01" CHECK (("sso" <> 2 and "pscId" IS NULL) OR ("sso" = 2 AND "pscId" IS NOT NULL));

-- By default from UM 3.6 vSphere Standard will be under Standard Bundle, to switch to Starter Bundle a script is used
UPDATE "ProductInBundle" SET "bundleId"=7 where "bundleId"=6 and "productId"=3;

-- Remove any selections of reporting NSX-base/NSX-Adv standalone
delete from "SeparateProduct" where "productId" in (6, 7);

alter table "VcSrm" add column "srmHostnames" varchar(512) not null default('');
update "VcSrm" set "srmHostnames" ="srmHostname";
alter table "VcSrm" drop column "srmHostname";
alter table "VcSrm" add column "updating" boolean not null default(false);

create function hosts_license_summary(vcServer int, report_starttime timestamp, report_endtime timestamp)
  returns table("licenseName" varchar, "licenseKey" varchar, "billingCategory" varchar)
  as $BODY$
select distinct l."name", l."code", lc."name"
from "Host" h, "HostHistory" hh, "License" l, "LicenseCategory" lc
where h."vcServerId" = vcServer and h."id" = hh."hostId"
and   (hh."time", hh."nextTime") overlaps (report_starttime, report_endtime)
and   hh."licenseId" = l."id" and hh."licenseCategoryId" = lc."id"
$BODY$
  language sql;

DROP FUNCTION vsan_space_usage(timestamp without time zone,timestamp without time zone,integer);
drop table "ClusterVsanHistoryBkup";

DROP FUNCTION migrate_vsan_history(interval, timestamp);
DROP FUNCTION get_clusters();
DROP FUNCTION create_cluster_sequence();
DROP TYPE vsanhist;
DROP TYPE vcclusters;
