#!/bin/bash

set -x

echo "Starting first-boot.sh"
source `dirname $0`/umsetenv.sh

# Stores the SSH/root enable flag in a temporary variable to be retrieved later.
# This is because the ovfenv vars will be wiped during db stuff
storeFlags() {
  set +x
  SSH_ENABLED=`ovfenv --key enable_ssh`
  SSH_ENABLED=$(echo "$SSH_ENABLED" | tr '[:upper:]' '[:lower:]')

  SUDO_OPEN=`ovfenv --key enable_open_sudo`
  SUDO_OPEN=$(echo "$SUDO_OPEN" | tr '[:upper:]' '[:lower:]')

  SSH_ROOT_ENABLED=`ovfenv --key enable_root_ssh`
  SSH_ROOT_ENABLED=$(echo "$SSH_ROOT_ENABLED" | tr '[:upper:]' '[:lower:]')
  set -x
}

setUpJre() {
   jresl=/usr/java/latest
   if [ ! -h $jresl ]; then
      ln -s /usr/java/jre-vmware/ $jresl
   fi
}


generateCertificate(){
    echo "Generating self-signed SSL certificate for HTTPS."
    $JAVA_HOME/bin/keytool -genkeypair -keyalg RSA -alias um -validity 1095 -keystore $TOMCAT_HOME/conf/webserver_certificate.jks -storepass silverpen -dname "CN=vUM, OU=Cloud, O=VMware Inc, L=Palo Alto, ST=California, C=US" -keypass silverpen -keysize 2048 2>&1 >/dev/null
    chmod 600 $TOMCAT_HOME/conf/webserver_certificate.jks
}

setupTomcat() {
    # we need to register service totally by ourself
    cp $optDir/tomcat/tomcat.service /usr/lib/systemd/system/tomcat.service
    chmod 644 /usr/lib/systemd/system/tomcat.service

    # tomcat config file
    cp $optDir/tomcat/tomcat.conf $TOMCAT_HOME/conf/tomcat.conf
    chmod 600 $TOMCAT_HOME/conf/tomcat.conf
    # Update config with memory limit - 2/3 of total available memory
    MAX_HEAP_KB=`awk '/MemTotal/ {print int($2*2/3)}' /proc/meminfo`
    echo "JAVA_OPTS=\"-server -Xms256M -Xmx${MAX_HEAP_KB}k -XX:MaxMetaspaceSize=384M -Xss512K -XX:+UseG1GC -XX:ParallelGCThreads=4 -XX:ConcGCThreads=4\"" >> $TOMCAT_HOME/conf/tomcat.conf

    # Tomcat by default have servlets for management/docs. We don't want them, will be deleted
    rm -rf $TOMCAT_HOME/webapps/*

    echo "Creating Usage Meter tcServer instance."
    cp $optDir/tomcat/server.xml $TOMCAT_HOME/conf

    cp $optDir/dist/usage-meter-webapp*.war $TOMCAT_HOME/webapps/um.war
    mkdir /var/log/usgmtr

    # tomcat need to increase file limits for himself (bug in SUSE, usgmtr limits will be be applied)
    # Currently this feature is not critical, but I want to keep it.
    # Note 50000 is a very big number, 4000 is killing the system. Actual number is in umhealth script.
    # tail --pid is used to wait forever.
    sed -i 's/echo "Tomcat started."/ sudo \/usr\/bin\/prlimit --nofile=50000 --pid `cat "$CATALINA_PID"`\n\n    echo "Tomcat started."\n tail --pid=`cat "$CATALINA_PID"` -f \/dev\/null /' $TOMCAT_HOME/bin/catalina.sh

    # cache directory for catalina
    mkdir -p /var/cache/tomcat/temp
    chown -R usgmtr:wheel /var/cache/tomcat
}

configPostgreSql() {
    ## If the database usgmtr exists, exit from the postgres initializaton process
    USGMTR_EXISTS=$(/opt/vmware/vpostgres/current/bin/psql -U postgres -tAc "SELECT 1 FROM pg_database WHERE datname='usgmtr'")
    RT=$?
    if [[ $RT -gt 0 ]]; then
       echo "vPostgres service is not up. Exit"
       return 0
    fi
    if [[ $USGMTR_EXISTS -eq 1 ]]; then
       echo "The database usgmtr exists. Exit"
       return 0
    fi
    chkconfig vpostgres 35
    service vpostgres stop
    RT=0
    COUNTER=10
    while [[ $RT -eq 0 && $COUNTER -gt 0 ]]; do
      sleep 1
      /opt/vmware/vpostgres/current/bin/psql -U postgres -c "select current_date" >  /dev/null 2> /dev/null
      RT=$?
      let COUNTER-=1
    done
    /usr/bin/rm -rf /var/vmware/vpostgres/current/pgdata/*
	/usr/bin/rm /var/vmware/vpostgres/current/init_db.log
	/usr/bin/rm /var/vmware/vpostgres/current/init_db.errs
	/usr/bin/rm /var/vmware/vpostgres/current/init_cert.log
	/usr/bin/rm /var/vmware/vpostgres/current/init_cert.errs
    su - postgres -c "/opt/vmware/vpostgres/current/bin/initdb --auth-local=trust --auth-host=md5 -E UTF8 --locale en_US.UTF-8 -D /var/vmware/vpostgres/current/pgdata/"
    cp $optDir/scripts/ovfEnv.xml /opt/vmware/etc/vami/ovfEnv.xml
    su - postgres -c "/opt/vmware/vpostgres/9.5/scripts/init_cert.sh > /var/vmware/vpostgres/current/init_cert.log 2>/var/vmware/vpostgres/current/init_cert.errs"
    su - postgres -c "cp $optDir/scripts/pg_hba.conf     /var/vmware/vpostgres/current/pgdata/pg_hba.conf"
    su - postgres -c "cp $optDir/scripts/postgresql.conf /var/vmware/vpostgres/current/pgdata/postgresql.conf"
    service vpostgres start
    RT=1
    COUNTER=10
    while [[ $RT -gt 0 && $COUNTER -gt 0 ]]; do
      sleep 1
      /opt/vmware/vpostgres/current/bin/psql -U postgres -c "select current_date" >  /dev/null 2> /dev/null
      RT=$?
      let COUNTER-=1
    done
    su - postgres -c "/opt/vmware/vpostgres/current/bin/createuser usgmtr"
    su - postgres -c "/opt/vmware/vpostgres/current/bin/createdb -O usgmtr usgmtr"
    su - usgmtr   -c "/opt/vmware/vpostgres/current/bin/psql -U usgmtr -f $optDir/database/postgresql-create.sql usgmtr"
	return 0
}

generateUUID() {
    echo "Generating UM instance unique identifier."
    chmod 700 $optDir/scripts/generate-um-uuid.sh
    $optDir/scripts/generate-um-uuid.sh
}

makeDirs() {
    mkdir $optDir/encryption
}

turnOffServices() {
    if [ "$SSH_ENABLED" = "true"  ]
    then
      echo "Starting sshd..."
      systemctl start sshd
    else
      echo "Stopping sshd..."
      systemctl disable sshd
      systemctl stop sshd
    fi
    # Stop vami because we don't use it, and there are security issues logged against its ports (PR 1572959)
    systemctl stop vami-sfcb
    systemctl stop vami-lighttp
    systemctl disable vami-sfcb
    systemctl disable vami-lighttp
    systemctl mask vami-sfcb
    systemctl mask vami-lighttp
}

addToSudoers() {
    # Disable default access from fresh install
    sed -i 's/Defaults targetpw/# Defaults targetpw/g' /etc/sudoers
    sed -i 's/^ALL/# ALL/g' /etc/sudoers

    # Add special commands that require sudo
    # create-support-bundle requires sudo to read system log files for generating support bundles.
    echo "usgmtr ALL = NOPASSWD: /opt/vmware/cloudusagemetering/scripts/support/create-support-bundle" >> /etc/sudoers

    # /usr/bin/prlimit used to increase number of max open files for tomcat. Becasue it starts as a service (bug/feature in SUSE)
    #    limits will not be applied. Because of this tomcat service script is updated with that. Since number of open files
    #    can be updated by root only, we need sudo access to prlimit command
    echo "usgmtr ALL = NOPASSWD: /usr/bin/prlimit" >> /etc/sudoers

    # migrateum requires several accesses as well
    echo "usgmtr ALL = NOPASSWD: /usr/sbin/service tomcat start" >> /etc/sudoers
    echo "usgmtr ALL = NOPASSWD: /usr/sbin/service tomcat stop" >> /etc/sudoers
    echo "usgmtr ALL = (postgres) NOPASSWD: /bin/bash" >> /etc/sudoers # need to run serveral steps as postgres user
}

addToCrontab() {
    # umhealth must be run every 5 minutes for lightweight checking.
    echo "*/5 * * * * root /opt/vmware/cloudusagemetering/scripts/umhealth.sh" >> /etc/cron.d/umhealth
    # fixing security issue with umhealth.sh. It is running as root, let's protect it
    chown root:root /opt/vmware/cloudusagemetering/scripts/umhealth.sh
    chmod 750  /opt/vmware/cloudusagemetering/scripts/umhealth.sh
}

addRootToWheel() {
    usermod -a -G wheel root
}

allowSshWhenStarted() {
    echo "sshd: ALL : ALLOW" >> /etc/hosts.allow
}

allowTcpSyslog() {
    echo 'syslog-ng : 127.0.0.1 : ALLOW' >> /etc/hosts.allow
    echo 'syslog-ng : [::1] : ALLOW' >> /etc/hosts.allow
}

changePasswd() {
    echo "Changing root passwd"
    echo "root:`ovfenv --key password`" | chpasswd
    echo "Changing usgmtr passwd"
    echo "usgmtr:`ovfenv --key usgmtr_password`" | chpasswd
}

startNtpDaemon(){
    echo "Initializing NTP"
    systemctl enable ntpd
    systemctl restart ntpd
}

disableSshRootLogin() {
    if [ "$SSH_ROOT_ENABLED" != "true"  ]
    then
      echo "Disabling ssh root access."
      sed -i 's/^PermitRootLogin .*$/PermitRootLogin no/g' /etc/ssh/sshd_config
    else
      echo "Enabling ssh root access."
      sed -i 's/^PermitRootLogin .*$/PermitRootLogin yes/g' /etc/ssh/sshd_config
      systemctl restart sshd
    fi
}

setSshdBanner() {
    echo "VMware vCloud Usage Meter" > /etc/issue
}

setNoFileLimit() {
    cp /etc/security/limits.conf /etc/security/limits.conf.bak
    echo 'usgmtr         hard    nofile         50000' >> /etc/security/limits.conf
    echo 'usgmtr         soft    nofile         50000' >> /etc/security/limits.conf
}

generateLocalSaltAndKey() {
    /usr/bin/perl -f `dirname $0`/generate-random-salt-phrase.pl
    chmod 600 $TOMCAT_HOME/webapps/um/WEB-INF/classes/props/local.props*
}

filePermLockdown() {
    chmod -R 700 $optDir
    chmod -R go-rwx $TOMCAT_HOME/conf
}


setUpJre
startNtpDaemon

# create user usgmtr:wheel for tomcat server and UM
useradd usgmtr
usermod -G wheel usgmtr
chage -M -1 usgmtr
chsh -s /bin/bash usgmtr
mkdir /home/usgmtr/.ssh
chown usgmtr:users /home/usgmtr/.ssh
chmod 700 /home/usgmtr/.ssh
touch /home/usgmtr/.ssh/known_hosts
chown usgmtr:users /home/usgmtr/.ssh/known_hosts
chmod 600 /home/usgmtr/.ssh/known_hosts

changePasswd
storeFlags
setNoFileLimit
makeDirs
setupTomcat
generateCertificate

chown -R usgmtr:wheel $optDir
chown -R usgmtr:wheel $TOMCAT_HOME /home/usgmtr /var/log/usgmtr

systemctl enable tomcat

configPostgreSql
generateUUID
turnOffServices
addToSudoers
addToCrontab
addRootToWheel
allowSshWhenStarted
allowTcpSyslog
disableSshRootLogin
setSshdBanner
filePermLockdown

# This is just a work-around that must be fixed for the next major release
echo "Fix for the VLSI warnings in catalina.out. The fix is adding a line in Tomcat's logging.properties."
echo "com.sun.xml.internal.ws.model.JavaMethodImpl.level=OFF" >> $TOMCAT_HOME/conf/logging.properties

systemctl start tomcat

# Need to wait for Tomcat to unpack the WAR file, then we can update the local properties.
# First, we wait until tomcat is started. Otherwise curl will return error right away.
# /usr/java/jre-vmware/bin/java -Djava.util.logging.config.file=/usr/share/tomcat/conf/logging.properties -Djava.util.logging.manager=org.apache.juli.ClassLoaderLogManager -server -Xms256M -Xmx2368248k -XX:MaxPermSize=256M -Xss512K -Djdk.tls.ephemeralDHKeySize=2048 -Djava.protocol.handler.pkgs=org.apache.catalina.webresources -classpath /usr/share/tomcat/bin/bootstrap.jar:/usr/share/tomcat/bin/tomcat-juli.jar -Dcatalina.base=/usr/share/tomcat -Dcatalina.home=/usr/share/tomcat -Djava.io.tmpdir=/var/cache/tomcat/temp org.apache.catalina.startup.Bootstrap start
until ps ax|grep "/usr/java/jre-vmware/bin/java" | grep "-server" |grep -v grep ; do sleep 1 ; done

# Then we wait until our unpacking is done and responding before updating local properties.
until curl -k --silent --output /dev/null https://127.0.0.1:8443/um/api/health ; do sleep 1 ; done

generateLocalSaltAndKey
systemctl restart tomcat



